<?php
/**
 * Display Functions
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Display
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

/**
 * Validates the vanity coupon code from the ajax call.
 * If successful, it's added to the vanity codes database for review.
 *
 * @since 1.0
 */
function validate_vanity_code() {
	// Verify nonce.
	if ( ! isset( $_REQUEST['vccNonce'] ) || false === wp_verify_nonce( $_REQUEST['vccNonce'], 'request_vanity_coupon_code' ) ) {
		wp_send_json_error(
			array(
				'vccError' => __( 'You do not have permission to initiate this request. Contact an administrator for more information.', 'affiliatewp-vanity-coupon-codes' ),
			)
		);
	}

	// Get data.
	$affiliate_id = isset( $_REQUEST['affiliateID'] ) ? sanitize_text_field( $_REQUEST['affiliateID'] ) : false;
	$coupon_id    = isset( $_REQUEST['couponID'] )    ? sanitize_text_field( $_REQUEST['couponID'] )    : false;
	$old_code     = isset( $_REQUEST['oldCode'] )     ? sanitize_text_field( $_REQUEST['oldCode'] )     : false;
	$new_code     = isset( $_REQUEST['newCode'] )     ? sanitize_text_field( $_REQUEST['newCode'] )     : false;
	$integration  = isset( $_REQUEST['integration'] ) ? sanitize_text_field( $_REQUEST['integration'] ) : false;
	$type         = isset( $_REQUEST['type'] )        ? sanitize_text_field( $_REQUEST['type'] )        : false;

	// Bail if coupon id already has a pending vanity coupon.
	$pending_code = affiliatewp_vanity_coupon_codes()->db->is_pending( $coupon_id );

	if ( is_object( $pending_code ) ) {
		wp_send_json_error( array( 'vccError' => __( 'A coupon code is already pending. You will have to wait until the current coupon code has been approved or rejected to submit another.', 'affiliatewp-vanity-coupon-codes' ) ) );
	}

	// Bail if missing data.
	if ( ! $affiliate_id || ! $coupon_id || ! $old_code || ! $integration || ! $type ) {
		wp_send_json_error( array( 'vccError' => __( 'Some data was missing from the request. Please try again.', 'affiliatewp-vanity-coupon-codes' ) ) );
	}

	// Bail if input is blank.
	if ( ! $new_code ) {
		wp_send_json_error( array( 'vccError' => __( 'New coupon code cannot be blank. Please try again.', 'affiliatewp-vanity-coupon-codes' ) ) );
	}

	// Get sanitized version of the new code.
	$sanitized_new_code = affiliatewp_vanity_coupon_codes()->db->sanitize_vanity_code( $new_code );

	// Bail if input includes invalid characters.
	if ( strtoupper( $new_code ) !== $sanitized_new_code ) {
		// If the sanitized version is potentially acceptable, share that code as an option in the error.
		if ( ! empty( $sanitized_new_code ) ) {
			wp_send_json_error( array( 'vccError' =>
				/* translators: Sanitized vanity coupon code string */
				sprintf( __( 'This coupon code contains invalid characters. You might try this instead: %s', 'affiliatewp-vanity-coupon-codes' ), $sanitized_new_code ) ) );
		} else {
			// If only invalid characters are used, ask them to try again.
			wp_send_json_error( array( 'vccError' => __( 'This coupon code contains invalid characters.', 'affiliatewp-vanity-coupon-codes' ) ) );
		}
	}

	// Bail if the code is not unique for the given integration.
	$is_unique = affiliatewp_vanity_coupon_codes()->db->is_unique( $sanitized_new_code, $integration );

	if ( ! $is_unique ) {
		wp_send_json_error( array( 'vccError' => __( 'Sorry, this coupon code is not available. Please try another.', 'affiliatewp-vanity-coupon-codes' ) ) );
	}

	// Bail if the code is over the char limit 191.
	$max_char_limit_for_coupon = 191;

	if ( strlen( $sanitized_new_code ) > $max_char_limit_for_coupon ) {
		wp_send_json_error( array( 'vccError' =>
			/* translators: Max character limit for the coupon code */
			sprintf( __( 'This coupon code must be %s characters or less.', 'affiliatewp-vanity-coupon-codes' ), $max_char_limit_for_coupon ) ) );
	}

	// Everything checks out, add to pending vanity codes table.
	affiliatewp_vanity_coupon_codes()->db->add( array(
		'affiliate_id' => $affiliate_id,
		'coupon_id'    => $coupon_id,
		'current_code' => $old_code,
		'vanity_code'  => $sanitized_new_code,
		'integration'  => $integration,
		'type'         => $type,
	) );

	wp_send_json_success( array( 'codePreview' =>
		/* translators: Sanitized vanity coupon code string */
		sprintf( __('Awaiting review: %s', 'affiliatewp-vanity-coupon-codes') , $sanitized_new_code ) ) );
}
add_action( 'wp_ajax_validate_vanity_code', 'validate_vanity_code' );