<?php
/**
 * Vanity Coupon Codes Display: Affiliate Area
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Display
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

namespace AffiliateWP_Vanity_Coupon_Codes\Display;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once AFFWP_VCC_PLUGIN_DIR . 'includes/display/functions.php';

/**
 * Vanity Coupon Codes Affiliate Area display class.
 *
 * @since 1.0
 *
 */
class Affiliate_Area_Display {
	/**
	 * Affiliate Area Display constructor.
	 *
	 * @since 1.0
	 */
	public function __construct() {
		add_filter( 'affwp_coupons_dashboard_code_td', array( $this, 'filter_coupon_cell' ), 10, 3 );
	}

	/**
	 * Filters the coupon code table cell content of the coupons table.
	 * Adds a form to request a vanity code and displays the submitted vanity code when pending review.
	 *
	 * @since 1.0
	 *
	 * @param string $coupon_code The coupon code.
	 * @param array $coupon {
	 *     Coupon details.
	 *
	 *     @type int     $id          Coupon ID.
	 *     @type sting   $type        Coupon type (manual or dynamic).
	 *     @type string  $code        Coupon code.
	 *     @type array   $amount      Coupon amount.
	 *     @type string  $integration Integration.
	 * }
	 * @param int $affiliate_id Affiliate ID.
	 */
	public function filter_coupon_cell( $coupon_code, $coupon, $affiliate_id ) {
		$pending = affiliatewp_vanity_coupon_codes()->db->is_pending( $coupon['id'] );
		ob_start();
		?>
		<div id="coupon-<?php echo esc_attr( $coupon['id'] ); ?>">
			<div class="affwp-vcc-edit-link-wrap">
				<?php echo $coupon_code ?> <small><a id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>" class="affwp-vcc-edit-link" href="#">(edit)</a></small>
			</div>

			<div id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>-form" class="affwp-vcc-coupon-form screen-reader-text">
				<?php wp_nonce_field( 'request_vanity_coupon_code', 'vanity_coupon_codes_nonce'); ?>
				<input type="text" id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>-input" name="vanity-coupon" data-id="<?php echo esc_attr( $coupon['id'] ); ?>" placeholder="<?php esc_attr_e( 'Enter a new coupon code', 'affiliatewp-vanity-coupon-codes' ) ?>" data-affiliate="<?php echo esc_attr( $affiliate_id ); ?>" data-code="<?php echo esc_attr( $coupon['code'] ); ?>" data-integration="<?php echo esc_attr( $coupon['integration'] ); ?>" data-type="<?php echo esc_attr( $coupon['type'] ); ?>" class="affwp-vcc-coupon-code">
				<input type="button" id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>-submit" class="affwp-vcc-submit button" value="<?php esc_attr_e( 'Request', 'vanity-coupon-codes' ); ?>">
			</div>
			<div id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>-error" class="affwp-vcc-error"></div>
			<div id="affwp-vcc-<?php echo esc_attr( $coupon['id'] ); ?>-pending" class="affwp-vcc-pending-code">
				<?php
					// Display the vanity codes awaiting review.
					if ( ! empty( $pending->vanity_code ) ) {
						/* translators: Vanity coupon code string */
						echo sprintf( __( 'Awaiting review: %s', 'affiliatewp-vanity-coupon-codes'), $pending->vanity_code );
					}
				?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}
}