<?php
/**
 * Vanity Codes Database
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

namespace AffiliateWP_Vanity_Coupon_Codes\Core;
use \Affiliate_WP_DB;
use AffiliateWP_Vanity_Coupon_Codes\Integrations;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Vanity Codes database class.
 *
 * @since 1.0
 *
 * @see Affiliate_WP_DB
 */
class Vanity_Codes_DB extends Affiliate_WP_DB {
	/**
	 * Cache group for queries.
	 *
	 * @internal DO NOT change. This is used externally both as a cache group and shortcut
	 *           for accessing db class instances via
	 *           affiliatewp_vanity_coupon_codes()->{$cache_group}->*.
	 *
	 * @since 1.0
	 * @var   string
	 */
	public $cache_group = 'vanity_codes';

	/**
	 * Get things started.
	 *
	 * @since 1.0
	 */
	public function __construct() {
		global $wpdb, $wp_version;

		if ( defined( 'AFFILIATE_WP_NETWORK_WIDE' ) && AFFILIATE_WP_NETWORK_WIDE ) {
			// Allows a single vanity coupon codes table for the whole network.
			$this->table_name  = 'affiliate_wp_vanity_coupon_codes';
		} else {
			$this->table_name  = $wpdb->prefix . 'affiliate_wp_vanity_coupon_codes';
		}
		$this->primary_key = 'vanity_code_id';
		$this->version     = '1.0';
		$this->create_table();
	}

	/**
	 * Defines the database columns and their default formats.
	 *
	 * @since 1.0
	 */
	public function get_columns() {
		return array(
			'vanity_code_id' => '%d',
			'coupon_id'      => '%d',
			'affiliate_id'   => '%d',
			'vanity_code'    => '%s',
			'type'           => '%s',
			'integration'    => '%s',
			'current_code'   => '%s',
		);
	}

	/**
	 * Return the number of results found for a given query.
	 *
	 * @since 1.0
	 *
	 * @param array $args Query arguments.
	 * @return int Number of results matching the query.
	 */
	public function count( $args = array() ) {
		return $this->get_vanity_codes( $args, true );
	}

	/**
	 * Adds a vanity coupon code.
	 *
	 * @since 1.0
	 *
	 * @param array $data {
	 *     Required. Data for adding a new vanity coupon code.
	 * 
	 *     @type int    $coupon_id    ID of the coupon this will change.
	 *     @type int    $affiliate_id Affiliate ID.
	 *     @type string $vanity_code  Vanity coupon code.
	 *     @type string $type         Coupon type (manual or dynamic.)
	 *     @type string $integration  Integration.
	 * }
	 * @return bool True if added, otherwise false.
	 */
	public function add( $data = array() ) {
		// Bail if data is empty or if it isn't an array.
		if ( empty( $data ) || ! is_array( $data ) ) {
			return false;
		}

		// Bail if the affiliate ID is invalid.
		if ( empty( $data['affiliate_id'] ) || false === affwp_get_affiliate( $data['affiliate_id'] ) ) {
			return false;
		}

		// Bail if any data is missing.
		if ( empty( $data['coupon_id'] ) || empty( $data['vanity_code'] ) || empty( $data['type'] ) || empty( $data['integration'] ) || empty( $data['current_code'] ) ) {
			return false;
		}

		// Bail if a coupon already has a pending vanity code.
		$pending_code = $this->is_pending( $data['coupon_id'] );
		if ( is_object( $pending_code ) ) {
			return false;
		}

		// Sanitize and store the vanity coupon code.
		$data['vanity_code'] = $this->sanitize_vanity_code( $data['vanity_code'] );

		$added = $this->insert( $data, 'vanity_coupon' );

		if ( $added ) {
			return true;
		}

		return false;
	}

	/**
	 * Get vanity coupon codes.
	 *
	 * @since 1.0
	 *
	 * @param array $args {
	 *     Optional. Arguments for querying vanity coupons. Default empty array.
	 *
	 *     @type int          $number           Number of vanity coupons to query for. Default 30.
	 *     @type int          $offset           Number of vanity coupons to offset the query for. Default 0.
	 *     @type int|array    $vanity_coupon_id Vanity Code ID or array of IDs to explicitly retrieve.
	 *                                          Default 0 (all).
	 *     @type int|array    $coupon_id        Coupon ID or array of IDs to explicitly retrieve. Default 0 (all).
	 *     @type int|array    $affiliate_id     Affiliate ID or array of IDs to explicitly retrieve. Default empty.
	 *     @type string|array $vanity_code      Vanity coupon code or array of vanity codes to explicitly
	 *                                          retrieve. Default empty.
	 *     @type string|array $current_code     Current coupon code or array of coupon codes to explicitly
	 *                                          retrieve. Default empty.
	 *     @type string|array $type             Coupon type, array of types, or empty for all. Default empty.
	 *     @type string|array $integration      Integration type, array of integrations, or empty for all.
	 *                                          Default empty.
	 *     @type string       $order            How to order returned results. Accepts 'ASC' or 'DESC'.
	 *                                          Default 'DESC'.
	 *     @type string       $orderby          Vanity coupons table column to order results by. Accepts any
	 *                                          affiliate_wp_vanity_coupon_codes field. Default 'vanity_code_id'.
	 *     @type string|array $fields           Specific fields to retrieve. Accepts 'ids', a single vanity
	 *                                          coupon field, or an array of fields. Default '*' (all).
	 * }
	 *
	 * @param bool $count Whether to retrieve only the total number of results found. Default false.
	 * @return array|int Array of coupon objects or field(s) (if found), int if `$count` is true.
	 */
	public function get_vanity_codes( $args = array(), $count = false  ) {
		global $wpdb;

		$defaults = array(
			'number'         => 30,
			'offset'         => 0,
			'vanity_code_id' => 0,
			'coupon_id'      => 0,
			'affiliate_id'   => 0,
			'vanity_code'    => '',
			'current_code'   => '',
			'type'           => '',
			'integration'    => '',
			'orderby'        => $this->primary_key,
			'order'          => 'ASC',
			'fields'         => '',
		);

		$args = wp_parse_args( $args, $defaults );

		if ( $args['number'] < 1 ) {
			$args['number'] = 999999999999;
		}

		$where = $join = '';

		// Specific vanity coupon code ids.
		if( ! empty( $args['vanity_code_id'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if( is_array( $args['vanity_code_id'] ) ) {
				$vanity_code_ids = implode( ',', array_map( 'intval', $args['vanity_code_id'] ) );
			} else {
				$vanity_code_ids = intval( $args['vanity_code_id'] );
			}

			$where .= "`vanity_code_id` IN( {$vanity_code_ids} ) ";
		}

		// Specific coupons ids.
		if( ! empty( $args['coupon_id'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if( is_array( $args['coupon_id'] ) ) {
				$coupon_ids = implode( ',', array_map( 'intval', $args['coupon_id'] ) );
			} else {
				$coupon_ids = intval( $args['coupon_id'] );
			}

			$where .= "`coupon_id` IN( {$coupon_ids} ) ";
		}

		// Specific affiliates.
		if ( ! empty( $args['affiliate_id'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if ( is_array( $args['affiliate_id'] ) ) {
				$affiliate_ids = implode( ',', array_map( 'intval', $args['affiliate_id'] ) );
			} else {
				$affiliate_ids = intval( $args['affiliate_id'] );
			}

			$where .= "`affiliate_id` IN( {$affiliate_ids} ) ";
		}

		// Specific vanity code or codes.
		if ( ! empty( $args['vanity_code'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if( is_array( $args['vanity_code'] ) ) {
				$where .= "`vanity_code` IN('" . implode( "','", array_map( $this->sanitize_vanity_code, $args['vanity_code'] ) ) . "') ";
			} else {
				$vanity_code = $this->sanitize_vanity_code( $args['vanity_code'] );
				$where .= "`vanity_code` = '" . $vanity_code . "' ";
			}
		}

		// Specific coupon code or codes.
		if ( ! empty( $args['current_code'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if( is_array( $args['current_code'] ) ) {
				$where .= "`current_code` IN('" . implode( "','", array_map( $this->sanitize_vanity_code, $args['current_code'] ) ) . "') ";
			} else {
				$current_code = $this->sanitize_vanity_code( $args['current_code'] );
				$where .= "`current_code` = '" . $current_code . "' ";
			}
		}

		// Specific coupon types.
		if ( ! empty( $args['type'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if ( is_array( $args['type'] ) ) {
				$where .= "`type` IN('" . implode( "','", array_map( 'sanitize_text_field', $args['type'] ) ) . "') ";
			} else {
				$type = sanitize_text_field( $args['type'] );
				$where .= "`type` = '" . $type . "' ";
			}
		}

		// Specific integrations.
		if ( ! empty( $args['integration'] ) ) {
			$where .= empty( $where ) ? "WHERE " : "AND ";

			if ( is_array( $args['integration'] ) ) {
				$where .= "`integration` IN('" . implode( "','", array_map( 'sanitize_text_field', $args['integration'] ) ) . "') ";
			} else {
				$integration = sanitize_text_field( $args['integration'] );
				$where .= "`integration` = '" . $integration . "' ";
			}
		}

		// Select valid vanity coupon codes only.
		$where .= empty( $where ) ? "WHERE " : "AND ";
		$where .= "`$this->primary_key` != ''";

		// There can be only two orders.
		if ( 'ASC' === strtoupper( $args['order'] ) ) {
			$order = 'ASC';
		} else {
			$order = 'DESC';
		}

		$orderby = array_key_exists( $args['orderby'], $this->get_columns() ) ? $args['orderby'] : $this->primary_key;

		// Overload args values for the benefit of the cache.
		$args['orderby'] = $orderby;
		$args['order']   = $order;

		// Fields.
		$callback = '';

		if ( 'ids' === $args['fields'] ) {
			$fields   = "$this->primary_key";
			$callback = 'intval';
		} else {
			$fields = $this->parse_fields( $args['fields'] );
		}

		$key = ( true === $count ) ? md5( 'affwp_vanity_codes_count' . serialize( $args ) ) : md5( 'affwp_vanity_codes_' . serialize( $args ) );

		$last_changed = wp_cache_get( 'last_changed', $this->cache_group );
		if ( ! $last_changed ) {
			$last_changed = microtime();
			wp_cache_set( 'last_changed', $last_changed, $this->cache_group );
		}

		$cache_key = "{$key}:{$last_changed}";

		$results = wp_cache_get( $cache_key, $this->cache_group );

		if ( false === $results ) {
			$clauses = compact( 'fields', 'join', 'where', 'orderby', 'order', 'count' );

			$results = $this->get_results( $clauses, $args, $callback );
		}

		wp_cache_add( $cache_key, $results, $this->cache_group, HOUR_IN_SECONDS );

		return $results;
	}

	/**
	 * Sanitizes the given vanity code.
	 *
	 * @since 1.0
	 *
	 * @param string $code Raw vanity code.
	 * @return string Sanitized vanity code.
	 */
	function sanitize_vanity_code( $code ) {
		// Remove special characters.
		$code = sanitize_key( $code );

		// Remove underscores.
		$code = str_replace( '_', '', $code );

		// Replace multiple hyphens with a single. For example if the affiliate has no first name.
		$code = preg_replace( '(-{2,})', '-', $code );

		// Remove hyphen from beginning and end.
		$code = trim( $code, '-' );

		// Return capitalized code.
		return strtoupper( $code );
	}

	/**
	 * Check if a given coupon ID has a pending vanity code.
	 *
	 * @since 1.0
	 * 
	 * @param int $coupon_id Coupon ID.
	 * @return bool|object Return pending vanity coupon object. Otherwise false.
	 */
	public function is_pending( $coupon_id ) {
		if ( empty( $coupon_id ) ) {
			return false;
		}
		// Check if this coupon id is already in the table.
		$coupon = $this->get_by( 'coupon_id', $coupon_id );
		return empty( $coupon ) ? false : $coupon;
	}

	/**
	 * Check if a given vanity code is unique in the vanity code table and it's integration.
	 *
	 * @since 1.0
	 *
	 * @param int    $coupon_code Coupon code.
	 * @param string $integration Integration.
	 * @return bool
	 */
	public function is_unique( $coupon_code, $integration ) {
		if ( empty( $coupon_code ) || empty( $integration ) ) {
			return false;
		}

		// Check that it's unique by the given integration.
		$integrations = new Integrations\Coupon_Integrations();
		$existing_code = $integrations->check_integration_for_existing_code( $coupon_code, $integration );

		// Also, check that it's not already pending in the vanity code table by that integration.
		$pending_code = $this->get_vanity_codes( array(
			'vanity_code' => $coupon_code,
			'integration' => $integration,
		) );

		return ( empty ( $existing_code ) && empty( $pending_code ) ) ? true : false;
	}

	/**
	 * Get vanity coupon by given vanity code ID.
	 *
	 * @since 1.0
	 *
	 * @param int $vanity_code_id Vanity code ID.
	 * @return bool|object Return vanity coupon object. Otherwise false.
	 */
	public function get_vanity_coupon( $vanity_code_id ) {
		if ( empty( $vanity_code_id ) ) {
			return false;
		}

		$coupon = $this->get_by( 'vanity_code_id', $vanity_code_id );
		return empty( $coupon ) ? false : $coupon;
	}

	/**
	 * Routine that creates the vanity coupon codes table.
	 *
	 * @since 1.0
	 */
	public function create_table() {
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		$sql = "CREATE TABLE {$this->table_name} (
			vanity_code_id bigint(20)   NOT NULL AUTO_INCREMENT,
			coupon_id      bigint(20)   NOT NULL,
			affiliate_id   bigint(20)   NOT NULL,
			vanity_code    varchar(191) NOT NULL,
			current_code   varchar(191) NOT NULL,
			type           tinytext     NOT NULL,
			integration    tinytext     NOT NULL,
			PRIMARY KEY (vanity_code_id),
			KEY vanity_code (vanity_code)
			) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;";

		dbDelta( $sql );

		update_option( $this->table_name . '_db_version', $this->version );
	}

}