<?php
/**
 * Vanity Coupon Codes Admin List Table
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

namespace AffiliateWP_Vanity_Coupon_Codes\Core;
use AffiliateWP_Vanity_Coupon_Codes\Integrations;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'AffWP\Admin\List_Table' ) ) {
	require_once AFFILIATEWP_PLUGIN_DIR . 'includes/abstracts/class-affwp-list-table.php';
}

/**
 * List Table class.
 *
 * Renders the vanity coupon codes awaiting review table.
 *
 * @since 1.0
 *
 * @see \AffWP\Admin\List_Table
 */
class List_Table extends \AffWP\Admin\List_Table {
	/**
	 * Default number of items to show per page.
	 *
	 * @var int
	 * @since 1.0
	 */
	public $per_page = 30;

	/**
	 * Total number of vanity coupon codes found.
	 *
	 * @var int
	 * @since 1.0
	 */
	public $total_count;

	/**
	 * Vanity Codes WP_List_Table object
	 * @var obj
	 * @since 1.0
	 */
	public $vanity_codes_table;


	/**
	 * Get things started.
	 *
	 * @access public
	 * @since  1.0
	 *
	 * @see WP_List_Table::__construct()
	 */
	public function __construct() {
		parent::__construct( array(
			'singular' => __( 'vanity-code', 'affiliatewp-vanity-coupon-codes' ),
			'plural'   => __( 'vanity-codes', 'affiliatewp-vanity-coupon-codes' ),
		) );

		// Set up total count.
		$this->get_vanity_code_count();
	}

	/**
	 * Retrieve the table columns.
	 *
	 * @access public
	 * @since 1.0
	 * @return array $columns Array of all the list table columns.
	 */
	public function get_columns() {
		$columns = array(
			'cb'           => '<input type="checkbox" />',
			'current_code' => __( 'Current coupon code', 'affiliatewp-vanity-coupon-codes' ),
			'vanity_code'  => __( 'New coupon code', 'affiliatewp-vanity-coupon-codes' ),
			'affiliate_id' => __( 'Affiliate', 'affiliatewp-vanity-coupon-codes' ),
			'actions'      => __( 'Actions', 'affiliatewp-vanity-coupon-codes' ),
		);

		/**
		 * Filters the vanity coupon codes list table columns.
		 *
		 * @since 1.0
		 *
		 * @param array                      $prepared_columns Prepared columns.
		 * @param array                      $columns          The columns for this list table.
		 * @param \Vanity_Coupon_Codes_Table $this             List table instance.
		 */
		return apply_filters( 'vanity_coupon_codes_table_columns', $this->prepare_columns( $columns ), $columns, $this );
	}

	/**
	 * Retrieve the table's sortable columns
	 *
	 * @access public
	 * @since 1.0
	 * @return array Array of all the sortable columns.
	 */
	public function get_sortable_columns() {
		$columns = array(
			'current_code' => array( 'current_code', false ),
			'vanity_code'  => array( 'vanity_code', false ),
			'affiliate_id' => array( 'affiliate_id', false ),
		);

		/**
		 * Filters the vanity coupon codes list table sortable columns.
		 *
		 * @since 1.0
		 *
		 * @param array                      $columns The sortable columns for this list table.
		 * @param \Vanity_Coupon_Codes_Table $this    List table instance.
		 */
		return apply_filters( 'vanity_coupon_codes_sortable_columns', $columns, $this );
	}

	/**
	 * This function renders most of the columns in the list table.
	 *
	 * @access public
	 * @since 1.0
	 *
	 * @param \AffiliateWP_Vanity_Coupon_Codes\Vanity_Code $vanity_code Contains all the vanity coupon code data.
	 * @param string                                       $column_name The name of the column.
	 * @return string Column Name.
	 */
	public function column_default( $vanity_code, $column_name ) {
		switch( $column_name ) {

			case 'affiliate_id':
				$value = isset( $vanity_code ) ? affwp_get_affiliate_username( $vanity_code->$column_name ) : '-';
				break;
			default:
				$value = isset( $vanity_code->$column_name ) ? $vanity_code->$column_name : '—';
				break;
		}

		/**
		 * Filters the default value for each column in the vanity coupon codes list table.
		 *
		 * This dynamic filter is appended with a suffix of the column name, for example:
		 *
		 *     `vanity_coupon_codes_table_current_code`
		 *
		 * @since 1.0
		 *
		 * @param string $value       Column data to show.
		 * @param array  $vanity_code Vanity coupon code data.
		 *
		 */
		return apply_filters( "vanity_coupon_codes_table_{$column_name}", $value, $vanity_code );
	}

	/**
	 * Render the checkbox column.
	 *
	 * @access public
	 * @since  1.0
	 *
	 * @param \AffiliateWP_Vanity_Coupon_Codes\Vanity_Code $vanity_code Contains all the vanity coupon code data
	 *                                                                  for the checkbox column.
	 * @return string Displays a checkbox.
	 */
	public function column_cb( $vanity_code ) {
		if ( ! $vanity_code ) {
			return '';
		}

		return '<input type="checkbox" name="vanity_code_id[]" value="' . absint( $vanity_code->vanity_code_id ) . '" />';
	}

	/**
	 * Render the actions column.
	 *
	 * @access public
	 * @since  1.0
	 *
	 * @param \AffiliateWP_Vanity_Coupon_Codes\Vanity_Code $vanity_code Contains all the data for the
	 *                                                                  actions column.
	 * @return string The actions HTML.
	 */
	public function column_actions( $vanity_code ) {
		if ( ! $vanity_code ) {
			return '—';
		}

		$row_actions = array();

		$base_query_args = array(
			'vanity_code_id' => $vanity_code->vanity_code_id
		);

		// Accept.
		$row_actions['accept'] = $this->get_row_action_link(
			__( 'Accept', 'affiliatewp-vanity-coupon-codes' ),
			array_merge( $base_query_args, array(
				'action' => 'accept'
			) ),
			array(
				'nonce' => 'vanity-coupon-codes-nonce',
				'class' => 'accept'
			)
		);

		// Reject.
		$row_actions['reject'] = $this->get_row_action_link(
			__( 'Reject', 'affiliatewp-vanity-coupon-codes' ),
			array_merge( $base_query_args, array(
				'action' => 'reject'
			) ),
			array(
				'nonce' => 'vanity-coupon-codes-nonce',
				'class' => 'reject'
			)
		);

		return $this->row_actions( $row_actions, true );
	}

	/**
	 * Renders the message to be displayed when there are no vanity coupon codes.
	 *
	 * @since 1.0
	 * @access public
	 */
	public function no_items() {
		_e( 'No vanity coupon codes are awaiting approval.', 'affiliatewp-vanity-coupon-codes' );
	}

	/**
	 * Retrieves the bulk actions.
	 *
	 * @access public
	 * @since  1.0
	 *
	 * @return array $actions The array of bulk actions.
	 */
	public function get_bulk_actions() {
		$actions = array(
			'accept' => __( 'Accept', 'affiliatewp-vanity-coupon-codes' ),
			'reject' => __( 'Reject', 'affiliatewp-vanity-coupon-codes' ),
		);

		/**
		 * Filters the bulk actions array for the vanity coupon codes list table.
		 *
		 * @since 1.0
		 *
		 * @param array $actions List of bulk actions.
		 */
		return apply_filters( 'vanity_coupon_codes_bulk_actions', $actions );
	}

	/**
	 * Processes bulk actions for the vanity coupon codes list table.
	 *
	 * @access public
	 * @since  1.0
	 */
	public function process_bulk_action() {
		if( empty( $_REQUEST['_wpnonce'] ) ) {
			return;
		}

		if( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-vanity-codes' ) && ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'vanity-coupon-codes-nonce' ) ) {
			return;
		}

		$ids = isset( $_GET['vanity_code_id'] ) ? $_GET['vanity_code_id'] : array();

		if ( ! is_array( $ids ) ) {
			$ids = array( $ids );
		}

		$ids    = array_map( 'intval', $ids );

		foreach ( $ids as $id ) {

			if ( 'reject' === $this->current_action() ) {
				// Send rejection email and delete from the table.
				$this->reject_vanity_code( $id );
			}

			if ( 'accept' === $this->current_action() ) {
				// Approve via the integration, email, and delete from the table.
				$this->approve_vanity_code( $id );
			}
		}

		// Refresh vanity code counts.
		$this->get_vanity_code_count();
	}

	/**
	 * Retrieve the vanity coupon code count.
	 *
	 * @access public
	 * @since 1.0
	 * @return void
	 */
	public function get_vanity_code_count() {
		$this->total_count = affiliatewp_vanity_coupon_codes()->db->count();
	}

	/**
	 * Retrieve all the data for all the Vanity Coupon Codes.
	 *
	 * @access public
	 * @since 1.0
	 * @return array $affiliate_data Array of all the data for the Vanity Coupon Codes.
	 */
	public function vanity_codes_data() {
		$page    = isset( $_GET['paged'] )   ? absint( $_GET['paged'] )        : 1;
		$status  = isset( $_GET['status'] )  ? sanitize_key( $_GET['status'] ) :     '';
		$order   = isset( $_GET['order'] )   ? $_GET['order']                  : 'DESC';
		$orderby = isset( $_GET['orderby'] ) ? $_GET['orderby']                : 'vanity_code_id';

		$per_page = $this->get_items_per_page( 'vanity_coupon_codes_per_page', $this->per_page );

		$args = wp_parse_args( $this->query_args, array(
			'number'  => $per_page,
			'offset'  => $per_page * ( $page - 1 ),
			'status'  => $status,
			'orderby' => sanitize_text_field( $orderby ),
			'order'   => sanitize_text_field( $order )
		) );

		$vanity_codes = affiliatewp_vanity_coupon_codes()->db->get_vanity_codes( $args );

		// Retrieve the "current" total count for pagination purposes.
		$args['number']      = -1;
		$this->current_count = affiliatewp_vanity_coupon_codes()->db->count( $args );

		return $vanity_codes;
	}

	/**
	 * Setup the final data for the table
	 *
	 * @access public
	 * @since 1.0
	 * @uses Vanity_Coupon_Codes_Table::get_columns()
	 * @uses Vanity_Coupon_Codes_Table::get_sortable_columns()
	 * @uses Vanity_Coupon_Codes_Table::process_bulk_action()
	 * @uses Vanity_Coupon_Codes_Table::vanity_codes_data()
	 * @uses WP_List_Table::get_pagenum()
	 * @uses WP_List_Table::set_pagination_args()
	 * @return void
	 */
	public function prepare_items() {
		$per_page = $this->get_items_per_page( 'vanity_coupon_codes_per_page', $this->per_page );

		$this->get_columns();

		$hidden = array();

		$this->get_sortable_columns();

		$this->get_column_info();

		$this->process_bulk_action();

		$this->items = $this->vanity_codes_data();

		$this->get_pagenum();

		$total_items = $this->current_count;

		$this->set_pagination_args( array(
			'total_items' => $total_items,
			'per_page'    => $per_page,
			'total_pages' => ceil( $total_items / $per_page )
		) );
	}

	/**
	 * Handle rejection of the vanity code.
	 * Sends rejection email and deletes from the vanity code table.
	 *
	 * @since 1.0
	 *
	 * @param int $vanity_code_id Vanity code id.
	 * @return void
	 */
	public function reject_vanity_code( $vanity_code_id = 0 ) {
		// Must be a valid vanity code id.
		if ( $vanity_code_id === 0 || empty( $vanity_code_id ) ) {
			return;
		}

		// Get vanity code data (affiliate_id and vanity code) for the email.
		$coupon = affiliatewp_vanity_coupon_codes()->db->get_vanity_coupon( $vanity_code_id );

		// Bail if the vanity coupon doesn't exist.
		if ( ! $coupon ){
			return;
		}

		/**
		 * Fires immediately after vanity code rejection.
		 *
		 * @since 1.0
		 *
		 * @param int    $affiliate_id Affiliate ID.
		 * @param string $vanity_code  Vanity Code
		 */
		do_action( 'affwp_vcc_send_reject_email', $coupon->affiliate_id, $coupon->vanity_code );

		// Delete the vanity coupon from the vanity code table.
		affiliatewp_vanity_coupon_codes()->db->delete( $vanity_code_id );
	}

	/**
	 * Handle approval of the vanity code.
	 * Updates the integration. Then if successful, sends approval email and deletes from the vanity code table.
	 *
	 * @since 1.0
	 *
	 * @param int $vanity_code_id Vanity code id.
	 * @return void
	 */
	public function approve_vanity_code( $vanity_code_id = 0 ) {
		// Must be a valid vanity code id.
		if ( $vanity_code_id === 0 || empty( $vanity_code_id ) ) {
			return;
		}

		// Get vanity code data.
		$coupon = affiliatewp_vanity_coupon_codes()->db->get_vanity_coupon( $vanity_code_id );

		// Bail if the vanity coupon doesn't exist.
		if ( ! $coupon ){
			return;
		}

		// Update coupon via integration.
		$integration = new Integrations\Coupon_Integrations();
		$updated = $integration->update_coupon( $coupon );

		// Only send email and delete if approved.
		if ( $updated ) {
			/**
			 * Fires immediately after vanity code approval.
			 *
			 * @since 1.0
			 *
			 * @param int    $affiliate_id Affiliate ID.
			 * @param string $vanity_code  Vanity Code
			 */
			do_action( 'affwp_vcc_send_approve_email', $coupon->affiliate_id, $coupon->vanity_code );

			// Delete the vanity coupon from the vanity code table.
			affiliatewp_vanity_coupon_codes()->db->delete( $vanity_code_id );
		}
	}

}