<?php
/**
 * Vanity Coupon Codes Admin
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */
namespace AffiliateWP_Vanity_Coupon_Codes\Core;
use \Affiliate_WP_Emails;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin class.
 * Registers submenu and handles sending emails.
 *
 * @since 1.0
 */
class Admin {
	/**
	 * Get things started.
	 *
	 * @access public
	 * @since  1.0
	 *
	 * @see WP_List_Table::__construct()
	 */
	public function __construct() {
		// Register Vanity Coupon Codes admin page as an admin page in AffiliateWP.
		add_filter( 'affwp_is_admin_page', array( $this, 'is_admin_page' ) );

		// Register menu.
		add_action( 'admin_menu', array( $this, 'register_submenu' ) );

		// Rejection emails.
		add_action( 'affwp_vcc_send_reject_email', array( $this, 'email_vanity_code_rejected_status' ), 10, 2 );

		// Approval emails.
		add_action( 'affwp_vcc_send_approve_email', array( $this, 'email_vanity_code_approval_status' ), 10, 2 );

		// Screen options.
		add_filter( 'set-screen-option', array( $this, 'affwp_vanity_coupon_codes_set_screen_option' ), 10, 3 );
	}

	/**
	 * Register Vanity Coupon Codes admin page as an admin page in AffiliateWP
	 *
	 * @access public
	 * @since 1.0
	 * @return array
	 */
	public function is_admin_page( $ret ) {

		if ( isset( $_GET['page'] ) && 'affiliate-wp-vanity-coupon-codes' === $_GET['page'] ) {
			$ret = true;
		}

		return $ret;

	}

	/**
	 * Register submenu page in admin.
	 *
	 * @since 1.0
	 */
	public function register_submenu() {
		$vanity_codes = add_submenu_page(
			'affiliate-wp',
			__( 'Vanity Coupon Codes',
			'affiliatewp-vanity-coupon-codes' ),
			__( 'Vanity Coupon Codes', 'affiliatewp-vanity-coupon-codes' ),
			'manage_affiliates',
			'affiliate-wp-vanity-coupon-codes',
			array( $this, 'affwp_vanity_coupon_codes_admin' ),
			13
		);
		add_action( 'load-' . $vanity_codes, array( $this, 'affwp_vanity_coupon_codes_screen_options' ) );
	}

	/**
	 * Add per page screen option to the Vanity Coupon Codes list table.
	 *
	 * @since 1.0
	 */
	public function affwp_vanity_coupon_codes_screen_options() {
		$screen = affwp_get_current_screen();
		if ( 'affiliate-wp-vanity-coupon-codes' !== $screen ) {
			return;
		}

		add_screen_option(
			'per_page',
			array(
				'label'   => __( 'Number of vanity coupon codes per page:', 'affiliatewp-vanity-coupon-codes' ),
				'option'  => 'vanity_coupon_codes_per_page',
				'default' => 30,
			)
		);

		// Instantiate the list table to make the columns array available to screen options.
		$this->vanity_codes_table = new List_Table();

		/**
		 * Fires in the screen-options area of the vanity coupons code screen.
		 *
		 * @since 1.0
		 *
		 * @param string $screen The current screen.
		 */
		do_action( 'affwp_vanity_coupon_codes_screen_options', $screen );
	}

	/**
	 * Renders per-page screen option value for the Vanity Coupon Codes list table.
	 *
	 * @since 1.0
	 *
	 * @param bool|int $status Screen option value.
	 * @param string   $option The option name.
	 * @param int      $value  The number of rows to use.
	 * @return bool|int Screen option value.
	 */
	public function affwp_vanity_coupon_codes_set_screen_option( $status, $option, $value ) {
		if ( 'vanity_coupon_codes_per_page' === $option ) {
			update_user_meta( get_current_user_id(), $option, $value );

			return $value;
		}

		return $status;
	}

	/**
	 * Loads the Vanity Codes admin screen.
	 *
	 * @since 1.0
	 */
	public function affwp_vanity_coupon_codes_admin() {
		$action = null;

		if ( isset( $_GET['action2'] ) && '-1' !== $_GET['action2'] ) {
			$action = $_GET['action2'];
		} elseif ( isset( $_GET['action'] ) && '-1' !== $_GET['action'] ) {
			$action = $_GET['action'];
		}

		$this->vanity_codes_table = new List_Table();
		$this->vanity_codes_table->prepare_items();
		?>
		<div class="wrap">
			<h1><?php _e( 'Vanity Coupon Codes', 'affiliatewp-vanity-coupon-codes' ); ?></h1>
			<form id="affwp-vanity-coupon-codes-filter" method="get" action="<?php echo admin_url( 'admin.php?page=affiliate-wp' ); ?>">
				<input type="hidden" name="page" value="affiliate-wp-vanity-coupon-codes" />
				<?php $this->vanity_codes_table->views() ?>
				<?php $this->vanity_codes_table->display() ?>
			</form>
		</div>
	<?php
	}

	/**
	 * Sends affiliate an email on vanity code rejection.
	 *
	 * @since 1.0
	 *
	 * @param int    $affiliate_id Affiliate ID.
	 * @param string $vanity_code  Vanity coupon code.
	 * @return void
	 */
	public function email_vanity_code_rejected_status( $affiliate_id, $vanity_code ) {
		// Set Affiliate ID to support email tags like {name}.
		$email = new Affiliate_WP_Emails();
		$email->__set( 'affiliate_id', $affiliate_id );

		// Get the given affiliate's email.
		$to = affwp_get_affiliate_email( $affiliate_id );

		// Subject.
		$subject = 'New Coupon Code Rejected';

		// Email body.
		$message = __( 'Hi {name},', 'affiliatewp-vanity-coupon-codes' ) . "\n\n";
		$message .= __( 'Sorry, your new coupon code name (' . $vanity_code . ') was rejected.', 'affiliatewp-vanity-coupon-codes' ) . "\n\n";

		// Send the rejection email.
		$email->send( $to, $subject, $message );
	}

	/**
	 * Sends affiliate an email on vanity code approval.
	 *
	 * @since 1.0
	 *
	 * @param int    $affiliate_id Affiliate ID.
	 * @param string $vanity_code  New vanity coupon code.
	 * @return void
	 */
	public function email_vanity_code_approval_status( $affiliate_id, $vanity_code ) {
		// Set Affiliate ID to support email tags like {name}.
		$email = new Affiliate_WP_Emails();
		$email->__set( 'affiliate_id', $affiliate_id );

		// Get the given affiliate's email.
		$to = affwp_get_affiliate_email( $affiliate_id );

		// Subject.
		$subject = 'New Coupon Code Accepted';

		// Email body.
		$message = __( 'Hi {name},', 'affiliatewp-vanity-coupon-codes' ) . "\n\n";
		$message .= __( 'Congratulations! Your new coupon code name (' . $vanity_code . ') was accepted.', 'affiliatewp-vanity-coupon-codes' ) . "\n\n";

		// Send the acceptance email.
		$email->send( $to, $subject, $message );
	}

}
