/* Affiliate Area Display Functionality */

jQuery(document).ready(function ($) {
	/**
	 * Removes the edit link and form from visual display when a vanity code is pending.
	 *
	 * @since 1.0
	 *
	 * @param {string} pendingCodeID The ID of the element where a pending code would be displayed.
	 *
	 * @return {void}
	 */
	const updateDisplayWhenPending = function(pendingCodeID){
		// Get the edit link ID.
		const vccEditID = pendingCodeID.replace('-pending', '');

		// Insert edit link.
		document.getElementById(vccEditID).innerText = affwp_vcc.editLink;

		if ( '' !== document.getElementById(pendingCodeID).innerText ) {
			// Get the form ID.
			const vccFormID = pendingCodeID.replace('-pending', '-form');
			// Only display edit link and form to screen readers.
			document.getElementById(vccEditID).classList.add('screen-reader-text');
			document.getElementById(vccFormID).classList.add('screen-reader-text');
		}
	}

	// Update display on page load.
	const pendingReviewCodes = $('.affwp-vcc-pending-code');
	for ( var c = 0; c < pendingReviewCodes.length; c++ ){
		updateDisplayWhenPending(pendingReviewCodes[c].id)
	}

	/**
	 * Handles what happens when edit/cancel link is clicked.
	 *
	 * This displays/hides the form to request a vanity coupon code and updates the edit/cancel text.
	 *
	 * @since 1.0
	 *
	 * @listens .affwp-vcc-edit-link~on:click
	 *
	 * @return {void}
	 */
	$('.affwp-vcc-edit-link').on('click', function(event) {
		event.preventDefault();
		// Use the link's ID to create the form ID selector.
		const vccFormID  = this.id + '-form';
		const editLinkID = this.id;

		// Get the form's classes. If it doesn't have `screen-reader-text`, it's currently displayed.
		var formDisplay = document.getElementById(vccFormID).classList;

		// If the form is displayed already, hide it. Otherwise, display it.
		if ( ! formDisplay.contains('screen-reader-text') ) {
			document.getElementById(editLinkID).innerText = affwp_vcc.editLink;
			document.getElementById(vccFormID).classList.add('screen-reader-text');
			// Get error ID and reset it to blank.
			const vccErrorID  = editLinkID + '-error';
			document.getElementById(vccErrorID).innerText = '';
		} else {
			// Select all edit links, forms, and errors.
			const vanityEditLinks = document.getElementsByClassName('affwp-vcc-edit-link');
			const vanityForms     = document.getElementsByClassName('affwp-vcc-coupon-form');
			const vccErrorMsgs    = document.getElementsByClassName('affwp-vcc-error');

			for ( var i = 0; i < vanityEditLinks.length; i++ ) {
				// Reset edit/cancel links to be 'edit'.
				vanityEditLinks[i].innerText = affwp_vcc.editLink;
				// Hide forms so only the one they clicked will be open.
				vanityForms[i].classList.add('screen-reader-text');
				// Reset error to blank.
				vccErrorMsgs[i].innerText = '';
			}

			// Change target link from 'edit' to 'cancel'.
			document.getElementById(editLinkID).innerText = affwp_vcc.cancelLink;
			// Open the target form.
			document.getElementById(vccFormID).classList.remove('screen-reader-text');
		}
	} );

	/**
	 * Handles what happens when the submit button is clicked.
	 *
	 * @since 1.0
	 *
	 * @listens .affwp-vcc-submit~on:click
	 *
	 * @return {void}
	 */
	$('.affwp-vcc-submit').on('click', function(event) {
		event.preventDefault();
		// Use the submit button's ID to create the input ID selector.
		var vccID = this.id.replace('-submit', '');
		const vccInputID = vccID + '-input';
		// Check the element with this ID exists. If so, check the new code.
		if ( document.getElementById(vccInputID) !== null ){
			checkNewCode(vccInputID);
		}
	} );

	/**
	 * Submits the requested vanity coupon code to be validated.
	 *
	 * @since 1.0
	 *
	 * @param {string} vccInputID The ID of the vanity coupon codes input.
	 *
	 * @return {void}
	 */
	const checkNewCode = function (vccInputID) {
		// Get the input by the given ID.
		const newVanityCode = document.getElementById(vccInputID);
		// Build the post data for the ajax call.
		const postData = {
			vccNonce:    document.getElementById('vanity_coupon_codes_nonce').value,
			action:     'validate_vanity_code',
			affiliateID: newVanityCode.getAttribute( 'data-affiliate' ),
			couponID:    newVanityCode.getAttribute( 'data-id' ),
			oldCode:     newVanityCode.getAttribute( 'data-code' ),
			newCode:     newVanityCode.value,
			integration: newVanityCode.getAttribute( 'data-integration' ),
			type:        newVanityCode.getAttribute( 'data-type' ),
		};

		jQuery.ajax({
			type: 'POST',
			url: affwp_vcc.ajaxurl,
			data: postData,
			dataType: "json",
			success: function ( response ) {
					// Get error ID.
					const vccErrorID  = vccInputID.replace('-input', '-error');

					if ( response.success && response.data ) {
						// Get the pending vanity code ID.
						const vccPendingID = vccInputID.replace('-input', '-pending');
						// Reset error to blank.
						document.getElementById(vccErrorID).innerText = '';
						// Display pending vanity code to table cell.
						document.getElementById(vccPendingID).innerText = response.data.codePreview;
						// Hide the form and edit link.
						updateDisplayWhenPending(vccPendingID);
					} else if ( response.success === false  && response.data ) {
						// Display error.
						const vccErrorMsg = response.data.vccError ? response.data.vccError : affwp_vcc.defaultError;
						document.getElementById(vccErrorID).innerText = vccErrorMsg;
					}
				},
			}).fail(function (data) {
				const vccErrorID  = vccInputID.replace('-input', '-error');
				document.getElementById(vccErrorID).innerText = affwp_vcc.defaultError;
		});	
	};
} );