<?php
/**
 * Script to create comprehensive fraud prevention demo data.
 *
 * Creates test data for ALL fraud scenarios using REAL detection flows:
 * - IP Velocity (affects Affiliates)
 * - PPC Traffic (affects Visits and Referrals)
 * - Self-Referral (affects Referrals)
 *
 * Usage: wp eval-file dev/create-test-referrals.php
 *
 * @package AffiliateWP
 * @since   2.31.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Creates comprehensive fraud prevention demo data.
 */
class Fraud_Demo_Data_Creator {

	/**
	 * Test prefix for cleanup.
	 */
	const TEST_PREFIX = '_fraud_demo_';

	/**
	 * Run all demo data creation.
	 */
	public function run() {
		WP_CLI::log( '' );
		WP_CLI::log( WP_CLI::colorize( '%G======================================================%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%G  FRAUD PREVENTION - COMPREHENSIVE DEMO DATA         %n' ) );
		WP_CLI::log( WP_CLI::colorize( '%G  Using REAL detection hooks and methods             %n' ) );
		WP_CLI::log( WP_CLI::colorize( '%G======================================================%n' ) );
		WP_CLI::log( '' );

		// Store original settings.
		$original_settings = $this->save_settings();

		// Create demo data for each feature.
		$this->create_ip_velocity_demos();
		$this->create_ppc_traffic_demos();
		$this->create_self_referral_demos();
		$this->create_normal_affiliate_demo();

		// Restore original settings.
		$this->restore_settings( $original_settings );

		$this->show_summary();
	}

	/**
	 * Save current fraud prevention settings.
	 */
	private function save_settings() {
		return [
			'ip_velocity'       => affiliate_wp()->settings->get( 'fraud_prevention_ip_velocity', 'allow' ),
			'ip_velocity_threshold' => affiliate_wp()->settings->get( 'fraud_prevention_ip_velocity_threshold', 3 ),
			'ppc_traffic'       => affiliate_wp()->settings->get( 'fraud_prevention_ppc_traffic', 'allow' ),
			'self_referrals'    => affiliate_wp()->settings->get( 'fraud_prevention_self_referrals', 'reject' ),
		];
	}

	/**
	 * Restore fraud prevention settings.
	 */
	private function restore_settings( $settings ) {
		affiliate_wp()->settings->set( [
			'fraud_prevention_ip_velocity'           => $settings['ip_velocity'],
			'fraud_prevention_ip_velocity_threshold' => $settings['ip_velocity_threshold'],
			'fraud_prevention_ppc_traffic'           => $settings['ppc_traffic'],
			'fraud_prevention_self_referrals'        => $settings['self_referrals'],
		], true );
	}

	/**
	 * Clean up filters between scenarios.
	 */
	private function cleanup_filters() {
		remove_all_filters( 'affwp_fraud_prevention_flag_ppc_traffic' );
		remove_all_filters( 'affwp_fraud_prevention_ppc_platform' );
		remove_all_filters( 'affwp_fraud_prevention_ppc_method' );
		remove_all_filters( 'affwp_fraud_prevention_flag_self_referral' );
		remove_all_filters( 'affwp_fraud_prevention_flag_referring_site' );
		$_GET = $_POST = $_REQUEST = [];
	}

	/**
	 * =========================================================================
	 * IP VELOCITY DEMOS (affects Affiliates)
	 * =========================================================================
	 */
	private function create_ip_velocity_demos() {
		WP_CLI::log( WP_CLI::colorize( '%B┌─────────────────────────────────────────────────────┐%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B│  IP VELOCITY - Affects AFFILIATES                  │%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B└─────────────────────────────────────────────────────┘%n' ) );
		WP_CLI::log( '' );

		// Set threshold to 2 for testing.
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity_threshold' => 2 ], true );

		// Demo 1: IP Velocity FLAG mode.
		WP_CLI::log( WP_CLI::colorize( '%Y► IP Velocity [FLAG mode]%n' ) );
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity' => 'flag' ], true );

		$ip_flag = '10.20.30.' . rand( 1, 254 );
		$_SERVER['REMOTE_ADDR'] = $ip_flag;

		for ( $i = 1; $i <= 3; $i++ ) {
			$aff_id = $this->create_affiliate( "ipvel_flag_{$i}", "IP Velocity Flag #{$i}" );
			$affiliate = affwp_get_affiliate( $aff_id );
			$has_flag = affwp_get_affiliate_meta( $aff_id, 'ip_velocity_flag', true );

			WP_CLI::log( sprintf(
				'  Affiliate #%d: status=%s, ip_velocity_flag=%s (IP: %s)',
				$aff_id,
				$affiliate->status,
				$has_flag ? 'YES' : 'no',
				$ip_flag
			) );
		}
		WP_CLI::log( '' );

		// Demo 2: IP Velocity REJECT mode.
		WP_CLI::log( WP_CLI::colorize( '%Y► IP Velocity [REJECT mode]%n' ) );
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity' => 'reject' ], true );

		$ip_reject = '10.20.31.' . rand( 1, 254 );
		$_SERVER['REMOTE_ADDR'] = $ip_reject;

		for ( $i = 1; $i <= 3; $i++ ) {
			$aff_id = $this->create_affiliate( "ipvel_reject_{$i}", "IP Velocity Reject #{$i}" );
			$affiliate = affwp_get_affiliate( $aff_id );
			$has_flag = affwp_get_affiliate_meta( $aff_id, 'ip_velocity_flag', true );

			WP_CLI::log( sprintf(
				'  Affiliate #%d: status=%s, ip_velocity_flag=%s (IP: %s)',
				$aff_id,
				$affiliate->status,
				$has_flag ? 'YES' : 'no',
				$ip_reject
			) );
		}
		WP_CLI::log( '' );

		// Reset IP.
		$_SERVER['REMOTE_ADDR'] = '127.0.0.1';
	}

	/**
	 * =========================================================================
	 * PPC TRAFFIC DEMOS (affects Visits and Referrals)
	 * =========================================================================
	 */
	private function create_ppc_traffic_demos() {
		WP_CLI::log( WP_CLI::colorize( '%B┌─────────────────────────────────────────────────────┐%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B│  PPC TRAFFIC - Affects VISITS and REFERRALS        │%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B└─────────────────────────────────────────────────────┘%n' ) );
		WP_CLI::log( '' );

		// Create an affiliate for PPC tests.
		$_SERVER['REMOTE_ADDR'] = '192.168.100.1';
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity' => 'allow' ], true );
		$ppc_affiliate_id = $this->create_affiliate( 'ppc_affiliate', 'PPC Test Affiliate' );
		WP_CLI::log( sprintf( '  Created affiliate #%d for PPC tests', $ppc_affiliate_id ) );
		WP_CLI::log( '' );

		// Demo 1: PPC FLAG mode - Google Ads.
		WP_CLI::log( WP_CLI::colorize( '%Y► PPC Traffic [FLAG mode] - Google Ads%n' ) );
		$this->cleanup_filters();
		affiliate_wp()->settings->set( [ 'fraud_prevention_ppc_traffic' => 'flag' ], true );

		$_REQUEST['gclid'] = 'CjwKCA_demo_google_' . time();
		$_GET['gclid'] = $_REQUEST['gclid'];

		// Trigger PPC detection.
		apply_filters( 'affwp_tracking_skip_track_visit', false, $ppc_affiliate_id, true, 'https://google.com' );

		// Create visit.
		$visit_data = apply_filters( 'affwp_pre_insert_visit_data', [
			'affiliate_id' => $ppc_affiliate_id,
			'url'          => home_url( '/product/?gclid=' . $_REQUEST['gclid'] ),
			'referrer'     => 'https://www.google.com/search',
			'ip'           => '192.168.50.' . rand( 1, 254 ),
		] );
		$visit_id = affiliate_wp()->visits->add( $visit_data );
		$visit = affwp_get_visit( $visit_id );

		// Create referral from this visit.
		$referral_id = $this->create_referral_through_filter( $ppc_affiliate_id, $visit_id, 89.99, 'Google Ads customer purchase' );
		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, $visit->flag ?: '(none)' ) );
		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );

		// Demo 2: PPC FLAG mode - Facebook Ads.
		WP_CLI::log( WP_CLI::colorize( '%Y► PPC Traffic [FLAG mode] - Facebook Ads%n' ) );
		$this->cleanup_filters();

		$_REQUEST['fbclid'] = 'IwAR_demo_facebook_' . time();
		$_GET['fbclid'] = $_REQUEST['fbclid'];

		apply_filters( 'affwp_tracking_skip_track_visit', false, $ppc_affiliate_id, true, 'https://facebook.com' );

		$visit_data = apply_filters( 'affwp_pre_insert_visit_data', [
			'affiliate_id' => $ppc_affiliate_id,
			'url'          => home_url( '/product/?fbclid=' . $_REQUEST['fbclid'] ),
			'referrer'     => 'https://www.facebook.com',
			'ip'           => '192.168.51.' . rand( 1, 254 ),
		] );
		$visit_id = affiliate_wp()->visits->add( $visit_data );
		$visit = affwp_get_visit( $visit_id );

		$referral_id = $this->create_referral_through_filter( $ppc_affiliate_id, $visit_id, 45.00, 'Facebook Ads customer purchase' );
		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, $visit->flag ?: '(none)' ) );
		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );

		// Demo 3: PPC FLAG mode - UTM CPC.
		WP_CLI::log( WP_CLI::colorize( '%Y► PPC Traffic [FLAG mode] - UTM Medium CPC%n' ) );
		$this->cleanup_filters();

		$_REQUEST['utm_medium'] = 'cpc';
		$_REQUEST['utm_source'] = 'bing';
		$_GET['utm_medium'] = $_REQUEST['utm_medium'];
		$_GET['utm_source'] = $_REQUEST['utm_source'];

		apply_filters( 'affwp_tracking_skip_track_visit', false, $ppc_affiliate_id, true, 'https://bing.com' );

		$visit_data = apply_filters( 'affwp_pre_insert_visit_data', [
			'affiliate_id' => $ppc_affiliate_id,
			'url'          => home_url( '/product/?utm_medium=cpc&utm_source=bing' ),
			'referrer'     => 'https://www.bing.com/search',
			'ip'           => '192.168.52.' . rand( 1, 254 ),
		] );
		$visit_id = affiliate_wp()->visits->add( $visit_data );
		$visit = affwp_get_visit( $visit_id );

		$referral_id = $this->create_referral_through_filter( $ppc_affiliate_id, $visit_id, 67.50, 'Bing Ads (UTM CPC) purchase' );
		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, $visit->flag ?: '(none)' ) );
		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );

		// Demo 4: PPC REJECT mode (visit blocked).
		WP_CLI::log( WP_CLI::colorize( '%Y► PPC Traffic [REJECT mode] - Visit Blocked%n' ) );
		$this->cleanup_filters();
		affiliate_wp()->settings->set( [ 'fraud_prevention_ppc_traffic' => 'reject' ], true );

		$_REQUEST['gclid'] = 'CjwKCA_demo_blocked_' . time();
		$_GET['gclid'] = $_REQUEST['gclid'];

		$skip = apply_filters( 'affwp_tracking_skip_track_visit', false, $ppc_affiliate_id, true, 'https://google.com' );

		WP_CLI::log( sprintf( '  Visit blocked: %s (no visit or referral created)', $skip ? 'YES' : 'NO' ) );
		WP_CLI::log( '' );

		$this->cleanup_filters();
	}

	/**
	 * =========================================================================
	 * SELF-REFERRAL DEMOS (affects Referrals)
	 * =========================================================================
	 */
	private function create_self_referral_demos() {
		WP_CLI::log( WP_CLI::colorize( '%B┌─────────────────────────────────────────────────────┐%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B│  SELF-REFERRAL - Affects REFERRALS                 │%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B└─────────────────────────────────────────────────────┘%n' ) );
		WP_CLI::log( '' );

		// Disable IP velocity for these tests.
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity' => 'allow' ], true );
		affiliate_wp()->settings->set( [ 'fraud_prevention_ppc_traffic' => 'allow' ], true );
		$_SERVER['REMOTE_ADDR'] = '192.168.200.1';

		// Demo 1: Self-Referral FLAG mode.
		WP_CLI::log( WP_CLI::colorize( '%Y► Self-Referral [FLAG mode]%n' ) );
		$this->cleanup_filters();
		affiliate_wp()->settings->set( [ 'fraud_prevention_self_referrals' => 'flag' ], true );

		$aff_id = $this->create_affiliate( 'selfref_flag', 'Self-Referral Flag Test' );
		$affiliate = affwp_get_affiliate( $aff_id );
		$user = get_user_by( 'id', $affiliate->user_id );

		WP_CLI::log( sprintf( '  Affiliate #%d (email: %s)', $aff_id, $user->user_email ) );

		// Create visit.
		$visit_id = affiliate_wp()->visits->add( [
			'affiliate_id' => $aff_id,
			'url'          => home_url( '/product/?ref=' . $aff_id ),
			'referrer'     => '',
			'ip'           => '192.168.60.' . rand( 1, 254 ),
		] );

		// Simulate self-referral check (email matches).
		apply_filters( 'affwp_is_customer_email_affiliate_email', true, $user->user_email, $aff_id );

		// Create referral.
		$referral_id = $this->create_referral_through_filter( $aff_id, $visit_id, 125.00, 'Affiliate self-purchase (flagged)' );
		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, affwp_get_visit( $visit_id )->flag ?: '(none)' ) );
		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );

		// Demo 2: Self-Referral REJECT mode.
		WP_CLI::log( WP_CLI::colorize( '%Y► Self-Referral [REJECT mode]%n' ) );
		$this->cleanup_filters();
		affiliate_wp()->settings->set( [ 'fraud_prevention_self_referrals' => 'reject' ], true );

		$_SERVER['REMOTE_ADDR'] = '192.168.201.1';
		$aff_id = $this->create_affiliate( 'selfref_reject', 'Self-Referral Reject Test' );
		$affiliate = affwp_get_affiliate( $aff_id );
		$user = get_user_by( 'id', $affiliate->user_id );

		WP_CLI::log( sprintf( '  Affiliate #%d (email: %s)', $aff_id, $user->user_email ) );

		// Create visit.
		$visit_id = affiliate_wp()->visits->add( [
			'affiliate_id' => $aff_id,
			'url'          => home_url( '/product/?ref=' . $aff_id ),
			'referrer'     => '',
			'ip'           => '192.168.61.' . rand( 1, 254 ),
		] );

		// Simulate self-referral check.
		$should_block = apply_filters( 'affwp_is_customer_email_affiliate_email', true, $user->user_email, $aff_id );

		// Create referral.
		$referral_id = $this->create_referral_through_filter( $aff_id, $visit_id, 95.00, 'Affiliate self-purchase (rejected)' );

		// In reject mode, integration rejects the referral.
		if ( $should_block ) {
			affwp_set_referral_status( $referral_id, 'rejected' );
		}

		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, affwp_get_visit( $visit_id )->flag ?: '(none)' ) );
		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );

		$this->cleanup_filters();
	}

	/**
	 * =========================================================================
	 * NORMAL AFFILIATE DEMO (no fraud)
	 * =========================================================================
	 */
	private function create_normal_affiliate_demo() {
		WP_CLI::log( WP_CLI::colorize( '%B┌─────────────────────────────────────────────────────┐%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B│  NORMAL - No fraud detected                        │%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%B└─────────────────────────────────────────────────────┘%n' ) );
		WP_CLI::log( '' );

		$this->cleanup_filters();
		affiliate_wp()->settings->set( [ 'fraud_prevention_ip_velocity' => 'allow' ], true );
		affiliate_wp()->settings->set( [ 'fraud_prevention_ppc_traffic' => 'allow' ], true );
		affiliate_wp()->settings->set( [ 'fraud_prevention_self_referrals' => 'reject' ], true );

		$_SERVER['REMOTE_ADDR'] = '192.168.250.1';

		$aff_id = $this->create_affiliate( 'normal', 'Legitimate Affiliate' );
		$affiliate = affwp_get_affiliate( $aff_id );

		WP_CLI::log( sprintf( '  Affiliate #%d: status=%s', $aff_id, $affiliate->status ) );

		// Create normal visit (no PPC).
		$visit_data = apply_filters( 'affwp_pre_insert_visit_data', [
			'affiliate_id' => $aff_id,
			'url'          => home_url( '/product/?ref=' . $aff_id ),
			'referrer'     => 'https://blog.example.com/review',
			'ip'           => '192.168.70.' . rand( 1, 254 ),
		] );
		$visit_id = affiliate_wp()->visits->add( $visit_data );
		$visit = affwp_get_visit( $visit_id );

		WP_CLI::log( sprintf( '  Visit #%d: flag=%s', $visit_id, $visit->flag ?: '(none)' ) );

		// Create normal referral (different customer, not self-referral).
		// Don't call the self-referral filter - this is a different customer.
		$referral_id = $this->create_referral_through_filter( $aff_id, $visit_id, 199.99, 'Legitimate customer purchase' );

		// Approve the referral.
		affwp_set_referral_status( $referral_id, 'unpaid' );
		$referral = affwp_get_referral( $referral_id );

		WP_CLI::log( sprintf( '  Referral #%d: status=%s, flag=%s', $referral_id, $referral->status, $referral->flag ?: '(none)' ) );
		WP_CLI::log( '' );
	}

	/**
	 * Create an affiliate through the real flow.
	 */
	private function create_affiliate( $suffix, $display_name ) {
		$username = self::TEST_PREFIX . $suffix . '_' . time() . rand( 100, 999 );
		$email    = $username . '@test.local';

		$user_id = wp_insert_user( [
			'user_login'   => $username,
			'user_email'   => $email,
			'user_pass'    => wp_generate_password(),
			'display_name' => $display_name,
		] );

		if ( is_wp_error( $user_id ) ) {
			WP_CLI::warning( 'Failed to create user: ' . $user_id->get_error_message() );
			return false;
		}

		// Create affiliate through REAL flow (triggers IP velocity check).
		return affwp_add_affiliate( [
			'user_id' => $user_id,
			'status'  => 'active',
		] );
	}

	/**
	 * Create a referral through the fraud detection filter.
	 */
	private function create_referral_through_filter( $affiliate_id, $visit_id, $amount, $description ) {
		$reference = 'order_' . time() . '_' . rand( 1000, 9999 );

		$args = [
			'affiliate_id' => $affiliate_id,
			'amount'       => $amount,
			'reference'    => $reference,
			'description'  => $description,
			'visit_id'     => $visit_id,
			'status'       => 'pending',
			'context'      => 'test',
		];

		// Apply the REAL fraud detection filter.
		$args = apply_filters(
			'affwp_insert_pending_referral',
			$args,
			$amount,
			$reference,
			$description,
			$affiliate_id,
			$visit_id,
			$args,
			'test'
		);

		return affiliate_wp()->referrals->add( $args );
	}

	/**
	 * Show summary of created data.
	 */
	private function show_summary() {
		global $wpdb;

		WP_CLI::log( WP_CLI::colorize( '%G======================================================%n' ) );
		WP_CLI::log( WP_CLI::colorize( '%G  SUMMARY                                            %n' ) );
		WP_CLI::log( WP_CLI::colorize( '%G======================================================%n' ) );
		WP_CLI::log( '' );

		// Count affiliates.
		$affiliates = $wpdb->get_results( "
			SELECT a.affiliate_id, a.status, u.display_name,
				   am.meta_value as ip_flag
			FROM " . affiliate_wp()->affiliates->table_name . " a
			JOIN {$wpdb->users} u ON a.user_id = u.ID
			LEFT JOIN " . affiliate_wp()->affiliate_meta->table_name . " am
				ON a.affiliate_id = am.affwp_affiliate_id AND am.meta_key = 'ip_velocity_flag'
			WHERE u.user_login LIKE '" . self::TEST_PREFIX . "%'
			ORDER BY a.affiliate_id
		" );

		WP_CLI::log( WP_CLI::colorize( '%YAffiliates:%n' ) );
		foreach ( $affiliates as $aff ) {
			$flag = $aff->ip_flag ? 'ip_velocity' : '-';
			WP_CLI::log( sprintf( '  #%d | %s | status: %s | flag: %s',
				$aff->affiliate_id,
				str_pad( $aff->display_name, 25 ),
				str_pad( $aff->status, 8 ),
				$flag
			) );
		}

		WP_CLI::log( '' );

		// Count visits with flags.
		$visits = $wpdb->get_results( "
			SELECT visit_id, affiliate_id, IFNULL(NULLIF(flag, ''), '-') as flag
			FROM " . affiliate_wp()->visits->table_name . "
			WHERE affiliate_id IN (
				SELECT affiliate_id FROM " . affiliate_wp()->affiliates->table_name . " a
				JOIN {$wpdb->users} u ON a.user_id = u.ID
				WHERE u.user_login LIKE '" . self::TEST_PREFIX . "%'
			)
			ORDER BY visit_id
		" );

		WP_CLI::log( WP_CLI::colorize( '%YVisits:%n' ) );
		foreach ( $visits as $visit ) {
			WP_CLI::log( sprintf( '  #%d | affiliate: %d | flag: %s',
				$visit->visit_id,
				$visit->affiliate_id,
				$visit->flag
			) );
		}

		WP_CLI::log( '' );

		// Count referrals with flags.
		$referrals = $wpdb->get_results( "
			SELECT referral_id, affiliate_id, amount, status,
				   IFNULL(NULLIF(flag, ''), '-') as flag, description
			FROM " . affiliate_wp()->referrals->table_name . "
			WHERE affiliate_id IN (
				SELECT affiliate_id FROM " . affiliate_wp()->affiliates->table_name . " a
				JOIN {$wpdb->users} u ON a.user_id = u.ID
				WHERE u.user_login LIKE '" . self::TEST_PREFIX . "%'
			)
			ORDER BY referral_id
		" );

		WP_CLI::log( WP_CLI::colorize( '%YReferrals:%n' ) );
		foreach ( $referrals as $ref ) {
			WP_CLI::log( sprintf( '  #%d | $%s | %s | flag: %s | %s',
				$ref->referral_id,
				str_pad( number_format( $ref->amount, 2 ), 7 ),
				str_pad( $ref->status, 8 ),
				str_pad( $ref->flag, 12 ),
				substr( $ref->description, 0, 35 )
			) );
		}

		WP_CLI::log( '' );
		WP_CLI::success( 'Demo data created successfully!' );
		WP_CLI::log( '' );
		WP_CLI::log( 'Admin URLs:' );
		WP_CLI::log( '  Affiliates: /wp-admin/admin.php?page=affiliate-wp-affiliates' );
		WP_CLI::log( '  Referrals:  /wp-admin/admin.php?page=affiliate-wp-referrals' );
		WP_CLI::log( '  Visits:     /wp-admin/admin.php?page=affiliate-wp-visits' );
		WP_CLI::log( '' );
		WP_CLI::log( 'To clean up: wp affwp fraud-test cleanup --yes' );
		WP_CLI::log( '' );
	}
}

// Run the demo data creator.
$creator = new Fraud_Demo_Data_Creator();
$creator->run();
