<?php

namespace GummiIO\AcfComponentField\Traits;

/**
 * UseV5Compatible Trait
 *
 * Trait that contains ACF version 5 compatible codes
 *
 * @since   2.0.5
 * @version 2.0.5
 */
trait UseV5Compatible
{
    /**
     * Check whether current ACF version is version 5 or not
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    protected function isV5()
    {
        return acf_component_field()->is_acf_version_5();
    }

    /**
     * Return the component settings in ACF version 5 format (HTML)
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    protected function loadComponentSettingsV5()
    {
        $ajax_error_format = '
            <tr class="acf-field" data-setting="component_field">
                <td class="acf-label"></td>
                <td class="acf-input acf-required"><span class="dashicons dashicons-warning"></span> %s</td>
            </tr>
        ';

        if (! acf_verify_ajax()) {
            printf($ajax_error_format, __('Session expired, please refresh the page and try again.', 'acf-component_field'));
            wp_die();
        }

        if (! $field_group_key = acf_maybe_get_POST('field_group_key')) {
            wp_die();
        }

        if (! $field_group = acf_get_field_group($field_group_key)) {
            printf($ajax_error_format, __('Unable to find the selected field group, please refresh the page and try again.', 'acf-component_field'));
            wp_die();
        }

        $group_default     = $this->defaults;
        $component_default = acf_maybe_get($field_group, 'acf_component_defaults');
        $value             = wp_parse_args($component_default, $group_default);

        foreach ($this->componentSettingFieldsV5() as $setting) {
            $setting['value'] = acf_maybe_get($value, $setting['name']);

            acf_render_field_setting([
                'type' => $this->name,
                'prefix' => acf_maybe_get_POST('prefix')
            ], $setting);
        }

        wp_die();
    }

    /**
     * Return the component setting fields in ACF version 5 format (flat array)
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    protected function componentSettingFieldsV5()
    {
        return [
            [
                'label'         => __('Repeatable', 'acf-component_field'),
                'instructions'  => __('Can this component be self repeated?', 'acf-component_field'),
                'type'          => 'true_false',
                'ui'             => true,
                'name'          => 'repeatable',
            ],
            [
                'label'         => __('Minimum Rows', 'acf'),
                'instructions'  => '',
                'type'          => 'number',
                'name'          => 'min',
                'placeholder'   => '0',
                'conditions'    => [
                    'field'    => 'repeatable',
                    'operator' => '==',
                    'value'    => 1
                ]
            ],
            [
                'label'         => __('Maximum Rows', 'acf'),
                'instructions'  => '',
                'type'          => 'number',
                'name'          => 'max',
                'placeholder'   => '0',
                'conditions'    => [
                    'field'    => 'repeatable',
                    'operator' => '==',
                    'value'    => 1
                ]
            ],
            [
                'label'         => __('Layout', 'acf'),
                'instructions'  => '',
                'class'         => 'acf-repeater-layout',
                'type'          => 'radio',
                'name'          => 'layout',
                'layout'        => 'horizontal',
                'choices'       => [
                    'table'     => __('Table', 'acf'),
                    'block'     => __('Block', 'acf'),
                    'row'       => __('Row', 'acf')
                ]
            ],
            [
                'label'         => __('Button Label', 'acf'),
                'instructions'  => '',
                'type'          => 'text',
                'name'          => 'button_label',
                'placeholder'    => __('Add Row', 'acf')
            ],
            [
                'label'         => __('Component Appearances', 'acf-component_field'),
                'instructions'  => sprintf(
                    '%s <br/>%s',
                    __('Alternative styles for component box.', 'acf-component_field'),
                    __('(usually used for non-repeating nesting components)', 'acf-component_field')
                ),
                'type'          => 'checkbox',
                'name'          => 'appearances',
                // 'layout'        => 'horizontal',
                'choices'       => [
                    'hide-outer-boundary' => __('Hide outer boundary', 'acf-component_field'),
                    'hide-outer-label'    => __('Hide outer field label', 'acf-component_field')
                ]
            ],
        ];
    }
}
