<?php

namespace GummiIO\AcfComponentField\Field;

use GummiIO\AcfComponentField\Field\loadComponentSettingsV6;
use GummiIO\AcfComponentField\Traits\UseV5Compatible;
use acf_field_repeater;

/**
 * Main Acf Component Field Class
 *
 * Component field class that extends the functionality from acf repeater field
 *
 * @since   2.0.0
 * @version 2.0.5
 */
class AcfComponentField extends acf_field_repeater
{
    /**
     * Version 5 backward compatibale related code
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    use UseV5Compatible;

	/**
	 * Setting up the field's property
	 *
     * @since   2.0.0
     * @version 2.0.5
	 */
	public function initialize()
	{
		$this->name     = 'component_field';
		$this->label    = __('Component Field', 'acf-component_field');
		$this->category = 'relational';
		$this->defaults = [
			'field_group_key' => null,
			'sub_fields'      => [],
			'repeatable'      => false,
			'min'             => null,
			'max'             => null,
            'rows_per_page'   => 20,
			'layout'          => 'block',
			'button_label'    => '',
			'collapsed'       => '',
			'appearances'     => []
		];

		add_action('wp_ajax_acf/field_types/component_field/load_settings', [$this, 'loadComponentSettings']);
		add_filter('acf/prepare_field/name=field_group_key', [$this, 'filterAvaliableFieldGroups']);

		$this->add_field_filter('acf/prepare_field_for_export', [$this, 'removeSubfieldsOnExport']);

        if (! $this->isV5()) {
            $this->add_filter('acf/pre_render_fields', [$this, 'pre_render_fields'], 10, 2);
            add_action('wp_ajax_acf/ajax/query_repeater', [$this, 'maybe_hijack_query_repeater_ajax'], 0);
        }
	}

	/**
	 * Enqueue addets and translation when on the field group edit screen
	 *
     * @since   2.0.0
     * @version 2.0.5
	 */
	public function field_group_admin_enqueue_scripts()
	{
		wp_enqueue_script(
			'acf-group-component_field',
			acf_component_field()->url('assets/js/field-group.js', true),
			['acf-pro-input'],
			acf_component_field()->version()
		);

		acf_localize_text([
			'convert_title'       => __('Convert field', 'acf-component_field'),
			'convert_text'        => __('Convert', 'acf-component_field'),
			'convert_warning'     => __('This field cannot be converted until its changes have been saved', 'acf-component_field'),
			'convert_popup_title' => __('Convert Field Group', 'acf-component_field'),
			'convert_delete_confirm_message' => __('Are you sure you want to delete the component afterwards?', 'acf-component_field'),
		]);

        if (! $this->isV5()) {
            wp_enqueue_style(
                'acf-input-component_field--group',
                acf_component_field()->url('assets/css/field-group.css', true),
                ['acf-pro-field-group'],
                acf_component_field()->version()
            );
        }
	}

	/**
	 * Enqueue addets on post that uses this acf fields
	 *
     * @since   2.0.0
     * @version 2.0.5
	 */
	public function input_admin_enqueue_scripts()
	{
		wp_enqueue_script(
			'acf-input-component_field--input',
			acf_component_field()->url('assets/js/field-input.js', true),
			['acf-pro-input'],
			acf_component_field()->version()
		);

		wp_enqueue_style(
			'acf-input-component_field--input',
			acf_component_field()->url('assets/css/field-input.css', true),
			['acf-pro-input'],
			acf_component_field()->version()
		);
	}

	/**
	 * Render field's settings. Additional settings will only be loaded if a
	 * selected field group is set and valid.
	 *
     * @since   2.0.0
     * @version 2.0.5
	 * @param   object $field acf field
	 */
	public function render_field_settings($field)
	{
		acf_render_field_setting($field, [
			'label'         => __('Field Group', 'acf-component_field'),
			'instructions'  => __('Select a field group to be used', 'acf-component_field'),
			'type'          => 'select',
			'name'          => 'field_group_key',
			'allow_null'    => true,
			'choices'       => [], // choices will be loaded from filter
			'acf-component_field::select_group' => 1,
		]);

        $field_group = acf_get_field_group($field_group_key = acf_maybe_get($field, 'field_group_key'));

		if (! $field_group_key || ! $field_group) {
			return;
		}

        if ($this->isV5()) {
            foreach ($this->componentSettingFieldsV5() as $setting) {
                acf_render_field_setting($field, $setting);
            }

            return;
        }

        $default_values = wp_parse_args(acf_maybe_get($field_group, 'acf_component_defaults'), $this->defaults);

		foreach ($this->componentSettingFields($field)['general'] as $setting) {
            $setting['value'] = isset($field[$setting['name']])? $field[$setting['name']] : acf_maybe_get($default_values, $setting['name']);
			acf_render_field_setting($field, $setting);
		}
	}

    /**
     * Renders the field settings used in the "Validation" tab. Only be loaded
     * if a selected field group is set and valid.
     *
     * @since   2.0.5
     * @version 2.0.5
     * @param   object $field acf field
     */
    public function render_field_validation_settings($field)
    {
        if (! $field_group_key = acf_maybe_get($field, 'field_group_key')) {
            return;
        }

        $field_group    = acf_get_field_group($field_group_key);
        $default_values = wp_parse_args(acf_maybe_get($field_group, 'acf_component_defaults'), $this->defaults);

        $field['min'] = empty($field['min'] ) ? '' : $field['min'];
        $field['max'] = empty($field['max'] ) ? '' : $field['max'];

        foreach ($this->componentSettingFields($field)['validation'] as $setting) {
            $setting['value'] = isset($field[$setting['name']])? $field[$setting['name']] : acf_maybe_get($default_values, $setting['name']);
            acf_render_field_setting($field, $setting);
        }
    }

    /**
     * Renders the field settings used in the "Presentation" tab. Only be loaded
     * if a selected field group is set and valid.
     *
     * @since   2.0.5
     * @version 2.0.5
     * @param   object $field acf field
     */
    public function render_field_presentation_settings($field)
    {
        if (! $field_group_key = acf_maybe_get($field, 'field_group_key')) {
            return;
        }

        $field_group    = acf_get_field_group($field_group_key);
        $default_values = wp_parse_args(acf_maybe_get($field_group, 'acf_component_defaults'), $this->defaults);

        foreach ($this->componentSettingFields($field)['presentation'] as $setting) {
            $setting['value'] = isset($field[$setting['name']])? $field[$setting['name']] : acf_maybe_get($default_values, $setting['name']);
            acf_render_field_setting($field, $setting);
        }
    }

	/**
	 * Do not delete the sub fields since they are referenced
	 *
     * @since   2.0.0
     * @version 2.0.0
	 * @param   object $field acf field
	 */
	public function delete_field($field)
	{
		return $field;
	}

	/**
	 * Do not duplicate the sub field since they are referenced
	 *
     * @since   2.0.0
     * @version 2.0.0
	 * @param   object $field acf field
	 */
	public function duplicate_field($field)
	{
		return $field;
	}

	/**
	 * Inject the selected field group's field as sub fields
	 *
     * @since   2.0.0
     * @version 2.0.5
	 * @param   object $field acf field
	 */
	public function load_field($field)
	{
        $field['min'] = (int) $field['min'];
        $field['max'] = (int) $field['max'];

        if ($this->isV5()) {
            $field['sub_fields'] = $this->fetchSubFields($field);
            return $field;
        }

		$field['sub_fields'] = array_map(function($sub_field) use ($field) {
            $sub_field['parent_repeater'] = $field['key'];
            return $sub_field;
        }, $this->fetchSubFields($field));

        if ('' === $field['button_label']) {
            $field['button_label'] = __('Add Row', 'acf');
        }

        if (empty($field['rows_per_page']) || (int) $field['rows_per_page'] < 1 ) {
            $field['rows_per_page'] = 20;
        }

		return $field;
	}

	/**
	 * Add appearance classes to the field's wrappepr. Adjust the min and max's
	 * value if repeatable is set to false
	 *
     * @since   2.0.0
     * @version 2.0.0
	 * @param   object $field acf field
	 */
	public function prepare_field($field)
	{
		$appearances = acf_maybe_get($field, 'appearances')? : [];

        foreach ($appearances as $appearance) {
            $field['wrapper']['class'] .= " acf-$appearance";
        }

		if (! $field['repeatable']) {
			$field['min'] = $field['max'] = 1;
		}

        return $field;
	}

	/**
	 * Show some error message if the field group cannot be found or not selected
	 *
     * @since   2.0.0
     * @version 2.0.0
	 * @param   object $field acf field
	 */
	public function render_field($field)
	{
		if (! $field['field_group_key']) {
			_e('A field group is not set on this component field.', 'acf-component_field');
			return;
		}

		if (! acf_get_field_group($field['field_group_key'])) {
			_e('Unable to locate the selected field group.', 'acf-component_field');
			return;
		}

		parent::render_field($field);
	}

	/**
	 * Ajax callbaack to output the additional component settings after a
	 * field group has been selected. The default values from the component
	 * field group will be used.
	 *
     * @since   2.0.0
     * @version 2.0.5
	 */
	public function loadComponentSettings()
	{
        if ($this->isV5()) {
            return $this->loadComponentSettingsV5();
        }

        $ajax_error_format = '
            <div class="acf-field" data-setting="component_field">
                <div class="acf-label"></div>
                <div class="acf-input acf-required"><span class="dashicons dashicons-warning"></span> %s</div>
            </div>
        ';

        $tabs = [
            'general'           => '',
            'validation'        => '',
            'presentation'      => '',
            'conditional_logic' => '',
        ];

        if (! acf_verify_ajax()) {
            $tabs['general'] = sprintf($ajax_error_format, __('Session expired, please refresh the page and try again.', 'acf-component_field'));
            wp_send_json_success($tabs);
        }

        if (! $field_group_key = acf_maybe_get_POST('field_group_key')) {
            wp_send_json_success($tabs);
        }

        if (! $field_group = acf_get_field_group($field_group_key)) {
            $tabs['general'] = sprintf($ajax_error_format, __('Unable to find the selected field group, please refresh the page and try again.', 'acf-component_field'));
            wp_send_json_success($tabs);
        }

        $field           = acf_maybe_get_POST('field_key');
        $field['prefix'] = acf_maybe_get_POST('prefix');
        $field           = acf_get_valid_field($field);

        $default_values  = wp_parse_args(acf_maybe_get($field_group, 'acf_component_defaults'), $this->defaults);

        foreach ($this->componentSettingFields($field) as $tab => $settings) {
            ob_start();

            foreach ($settings as $setting) {
                $setting['value'] = acf_maybe_get($default_values, $setting['name']);

                acf_render_field_setting([
                    'type' => $this->name,
                    'prefix' => acf_maybe_get_POST('prefix')
                ], $setting);
            }

            $tabs[$tab] = ob_get_clean();
        }

        wp_send_json_success($tabs);
	}

	/**
	 * Dynamically list the available component field groups to the select field
	 *
     * @since   2.0.0
     * @version 2.0.3
	 * @param   object $field acf field
	 */
	public function filterAvaliableFieldGroups($field)
	{
		if (acf_maybe_get($field, 'acf-component_field::select_group')) {
            $component_groups = acf_component_field('query')->getComponents();
            $field['choices'] = wp_list_pluck($component_groups, 'title', 'key');
		}

		return $field;
	}

	/**
	 * Do not include the referenced sub fields when exporting/duplicating the
	 * component field
	 *
     * @since   2.0.0
     * @version 2.0.0
	 * @param   object $field acf field
	 */
	public function removeSubfieldsOnExport($field)
	{
		$field['sub_fields'] = [];

		return $field;
	}

    /**
     * Maybe Hijack the ajax for repeater to get rows, so the _is_rending property
     * is set properly on the correct class
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    public function maybe_hijack_query_repeater_ajax()
    {
        if (! $field_key = acf_maybe_get_POST('field_key')) {
            return;
        }

        if (! $field = acf_get_field($field_key)) {
            return;
        }

        if ($field['type'] !== $this->name) {
            return;
        }

        remove_action('wp_ajax_acf/ajax/query_repeater', [acf_get_field_type('repeater'), 'ajax_get_rows']);
        add_action('wp_ajax_acf/ajax/query_repeater', [$this, 'ajax_get_rows']);
    }

	/**
	 * The additional field settings properties
	 *
     * @since   2.0.0
     * @version 2.0.5
	 */
	public function componentSettingFields($field = []) // @TODO ???
	{
        if ($this->isV5()) {
            return $this->componentSettingFieldsV5();
        }

        $tabs = [
            'general' => [
                [
                    'label'         => __('Layout', 'acf'),
                    'instructions'  => '',
                    'class'         => 'acf-repeater-layout',
                    'type'          => 'radio',
                    'name'          => 'layout',
                    'layout'        => 'horizontal',
                    'choices'       => [
                        'table'     => __('Table', 'acf'),
                        'block'     => __('Block', 'acf'),
                        'row'       => __('Row', 'acf')
                    ]
                ],
                [
                    'label'         => __('Repeatable', 'acf-component_field'),
                    'instructions'  => __('Can this component be self repeated?', 'acf-component_field'),
                    'type'          => 'true_false',
                    'ui'             => true,
                    'name'          => 'repeatable',
                ],
            ],
            'validation' => [
                [
                    'label'         => __('Minimum Rows', 'acf'),
                    'instructions'  => '',
                    'type'          => 'number',
                    'name'          => 'min',
                    'placeholder'   => '0',
                    'conditions'    => [
                        'field'    => 'repeatable',
                        'operator' => '==',
                        'value'    => 1
                    ]
                ],
                [
                    'label'         => __('Maximum Rows', 'acf'),
                    'instructions'  => '',
                    'type'          => 'number',
                    'name'          => 'max',
                    'placeholder'   => '0',
                    'conditions'    => [
                        'field'    => 'repeatable',
                        'operator' => '==',
                        'value'    => 1
                    ]
                ],
            ],
            'presentation' => [
                // Collapsed
                [
                    'label'         => __('Button Label', 'acf'),
                    'instructions'  => '',
                    'type'          => 'text',
                    'name'          => 'button_label',
                    'placeholder'    => __('Add Row', 'acf')
                ],
                [
                    'label'         => __('Component Appearances', 'acf-component_field'),
                    'instructions'  => esc_attr(sprintf(
                        '%s <br/>%s',
                        __('Alternative styles for component box.', 'acf-component_field'),
                        __('(usually used for non-repeating nesting components)', 'acf-component_field')
                    )),
                    'type'          => 'checkbox',
                    'name'          => 'appearances',
                    // 'layout'        => 'horizontal',
                    'choices'       => [
                        'hide-outer-boundary' => __('Hide outer boundary', 'acf-component_field'),
                        'hide-outer-label'    => __('Hide outer field label', 'acf-component_field')
                    ]
                ],
            ],
            'conditional_logic' => [],
        ];

        if (empty($field['parent_repeater']) && empty($field['parent_layout'])) {
            $tabs['general'][] = [
                'label'        => __('Pagination', 'acf'),
                'instructions' => __('Useful for fields with a large number of rows.', 'acf'),
                'class'        => 'acf-repeater-pagination',
                'type'         => 'true_false',
                'name'         => 'pagination',
                'ui'           => 1,
                'conditions'   => [
                    'field'    => 'repeatable',
                    'operator' => '==',
                    'value'    => 1
                ]
            ];

            $tabs['general'][] = [
                'label'        => __('Rows Per Page', 'acf'),
                'instructions' => __('Set the number of rows to be displayed on a page.', 'acf'),
                'class'        => 'acf-repeater-pagination-num-rows',
                'type'         => 'number',
                'name'         => 'rows_per_page',
                'placeholder'  => 20,
                'ui'           => 1,
                'min'          => 1,
                'conditions'   => [
                    'field'    => 'pagination',
                    'operator' => '==',
                    'value'    => 1
                ]
            ];
        }

        return $tabs;
	}

    /**
     * Return a flat array of available settings. ACF version 6 uses nested array
     *
     * @since   2.0.5
     * @version 2.0.5
     */
    public function componentAllSettingFields()
    {
        if ($this->isV5()) {
            return $this->componentSettingFieldsV5();
        }

        $fields = [];

        foreach ($this->componentSettingFields() as $settings) {
            foreach ($settings as $field) {
                $fields[] = $field;
            }
        }

        return $fields;
    }

	/**
	 * Fetch the givent field'd selected field group, and return its fields
	 *
     * @since   2.0.0
     * @version 2.0.3
	 * @param   object $field acf field
	 */
	protected function fetchSubFields($field)
	{
		$group_key = $field['field_group_key'];

		$field_group = acf_component_field('query')->getComponent($group_key);

		return acf_get_fields($field_group);
	}
}
