/* eslint no-undef: "off", no-alert: "off" */
( function( $ ) {
	'use strict';

	const apopToggleClose = $( '.apop-accessibility-plugin .apop-close' );
	const apopToggleOpen = $( '.apop-accessibility-plugin .apop-toggle' );
	const apopAccessibility = $( '.apop-accessibility-plugin .apop-accessibility' );
	const apopToggleLanguages = $( '.apop-accessibility-plugin .apop-languages' );
	const apopLanguageList = $( '.apop-accessibility-settings .apop-list-of-languages' );
	const apopSkipElements = '.apop-plugin-onetap, .apop-plugin-onetap *, .apop-toggle, .apop-toggle *, #wpadminbar, #wpadminbar *, rs-fullwidth-wrap, rs-fullwidth-wrap *, rs-module-wrap, rs-module-wrap *, sr7-module, sr7-module *';

	// Open Accessibility.
	apopToggleOpen.click( function( event ) {
		event.stopPropagation();
		apopAccessibility.removeClass( 'apop-toggle-close' ).addClass( 'apop-toggle-open' );
		apopToggleClose.show( 100 );
	} );

	// Close Accessibility.
	apopToggleClose.click( function( event ) {
		event.stopPropagation();
		apopAccessibility.removeClass( 'apop-toggle-open' ).addClass( 'apop-toggle-close' );
		apopToggleClose.hide( 100 );
	} );

	// Prevent auto-close when clicking inside accessibility panel.
	apopAccessibility.click( function( event ) {
		apopLanguageList.fadeOut( 350 );
		apopToggleLanguages.removeClass( 'apop-active' );
		if ( ! $( event.target ).closest( '.apop-reset-settings' ).length ) {
			event.stopPropagation();
		}
	} );

	// Toggle list of languages.
	apopToggleLanguages.click( function( event ) {
		event.stopPropagation();
		$( this ).toggleClass( 'apop-active' );
		apopLanguageList.fadeToggle( 350 );
	} );

	// Auto-close elements when clicking outside
	$( document ).click( function( event ) {
		const isClickInsideAccessibility = $( event.target ).closest( '.apop-accessibility' ).length > 0;
		const isClickInsideLanguages = $( event.target ).closest( '.apop-languages, .apop-list-of-languages' ).length > 0;

		// If clicking outside the accessibility panel, close accessibility
		if ( ! isClickInsideAccessibility ) {
			apopAccessibility.removeClass( 'apop-toggle-open' ).addClass( 'apop-toggle-close' );
			apopToggleClose.hide( 100 );
		}

		// If clicking outside the language list, close the language list
		if ( ! isClickInsideLanguages ) {
			apopLanguageList.fadeOut( 350 );
			apopToggleLanguages.removeClass( 'apop-active' );
		}
	} );

	// Get the current date
	const apopToday = new Date();

	// Extract the apopYear, apopMonth, and apopDay
	const apopYear = apopToday.getFullYear(); // Get the full apopYear (e.g., 2024)
	const apopMonth = String( apopToday.getMonth() + 1 ).padStart( 2, '0' ); // Get the apopMonth (0-11) and add 1; pad with 0 if needed
	const apopDay = String( apopToday.getDate() ).padStart( 2, '0' ); // Get the apopDay of the apopMonth (1-31) and pad with 0 if needed

	// Create a formatted date string for the start date in the format YYYY-MM-DD
	const apopStartDate = `${ apopYear }-${ apopMonth }-${ apopDay }`;

	// Create a new date object for the end date by adding 2 days to the current date
	const apopEndDateObject = new Date( apopToday ); // Create a new Date object based on apopToday
	apopEndDateObject.setDate( apopEndDateObject.getDate() + 2 ); // Add 2 days

	// Extract the year, month, and day for the end date
	const apopEndYear = apopEndDateObject.getFullYear();
	const apopEndMonth = String( apopEndDateObject.getMonth() + 1 ).padStart( 2, '0' );
	const apopEndDay = String( apopEndDateObject.getDate() ).padStart( 2, '0' );

	// Create a formatted date string for the end date
	const apopEndDate = `${ apopEndYear }-${ apopEndMonth }-${ apopEndDay }`;

	// console.log(apopStartDate); // Output the start date
	// console.log(apopEndDate);   // Output the end date

	// Default values for apopLocalStorage
	const apopProAccessibilityDefault = {
		dynamicFeatureSet: {
			visionImpairedMode: false,
			seizureSafeProfileMode: false,
			adhdFriendlyMode: false,
			blindnessMode: false,
			epilepsySafeMode: false,
		},
		activeBorders: {
			biggerText: 0,
			cursor: 0,
			lineHeight: 0,
			letterSpacing: 0,
			textAlign: 0,
			textMagnifier: 0,
			invertColors: 0,
			brightness: 0,
			contrast: 0,
			grayscale: 0,
			saturation: 0,
		},
		biggerText: false,
		cursor: false,
		lineHeight: false,
		letterSpacing: false,
		readableFont: false,
		dyslexicFont: false,
		textAlign: false,
		textMagnifier: false,
		highlightLinks: false,
		invertColors: false,
		brightness: false,
		contrast: false,
		grayscale: false,
		saturation: false,
		readingLine: false,
		keyboardNavigation: false,
		highlightTitles: false,
		readingMask: false,
		hideImages: false,
		highlightAll: false,
		readPage: false,
		muteSounds: false,
		stopAnimations: false,
		information: {
			updated: 'apop-version-17',
			language: apopAjaxObject.getSettings.language,
			developer: 'Yuky Hendiawan',
			startDate: apopStartDate,
			endDate: apopEndDate,
		},
	};

	// If 'apopLocalStorage' does not exist in localStorage, create it
	const apopLocalStorage = 'apop-accessibility-pro';
	if ( ! localStorage.getItem( apopLocalStorage ) ) {
		localStorage.setItem( apopLocalStorage, JSON.stringify( apopProAccessibilityDefault ) );
	} else {
		// Retrieve the existing data from localStorage
		const accessibilityData = JSON.parse( localStorage.getItem( apopLocalStorage ) );

		// Check if 'information.updated' exists and whether its value is 'apop-version-17'
		if ( typeof accessibilityData.information === 'undefined' ||
			typeof accessibilityData.information.updated === 'undefined' ||
			accessibilityData.information.updated !== 'apop-version-17' ) {
			localStorage.removeItem( apopLocalStorage );
			localStorage.setItem( apopLocalStorage, JSON.stringify( apopProAccessibilityDefault ) );
		}
	}

	// Retrieves accessibility data from local storage.
	function getDataAccessibilityData() {
		const accessibilityData = JSON.parse( localStorage.getItem( apopLocalStorage ) );
		return accessibilityData;
	}

	/**
	 * Updates the 'data-on' and 'data-off' attributes of elements with the class 'label-mode-switch-inner'.
	 * If the attribute value exceeds 3 characters, it truncates it and appends '...'.
	 */
	function updateLabelModeSwitch() {
		$( '.apop-accessibility .label-mode-switch-inner' ).each( function() {
			const $this = $( this );
			let dataOn = $this.attr( 'data-on' );
			let dataOff = $this.attr( 'data-off' );

			if ( dataOn.length > 3 ) {
				dataOn = dataOn.substring( 0, 3 ) + '.';
			}

			if ( dataOff.length > 3 ) {
				dataOff = dataOff.substring( 0, 3 ) + '.';
			}

			$this.attr( 'data-on', dataOn );
			$this.attr( 'data-off', dataOff );
		} );
	}

	// Updates the country flag based on the selected language.
	updateLanguageFlag();
	function updateLanguageFlag() {
		// Remove the 'apop-active' class from all country flag images
		$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img' ).removeClass( 'apop-active' );

		// Add the 'apop-active' class to the image with the alt attribute matching the selected language
		$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img[alt="' + getDataAccessibilityData().information.language + '"]' ).addClass( 'apop-active' );
	}

	// Event handler for language selection
	$( 'nav.apop-accessibility header.apop-header-top .apop-list-of-languages ul li' ).click( function() {
		const selectedLanguage = $( this ).attr( 'data-language' ); // Get the selected language from the data attribute
		const languageName = $( this ).text(); // Get the name of the selected language

		// Remove active class from the images
		$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img' ).removeClass( 'apop-active' );

		// Add active class from the images
		$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img[alt="' + selectedLanguage + '"]' ).addClass( 'apop-active' );

		// Remove active class from the language toggle
		$( apopToggleLanguages ).removeClass( 'apop-active' );

		// Update the displayed language name
		$( 'nav.apop-accessibility header.apop-header-top .apop-languages .apop-text span' ).text( languageName );

		// Update the header content based on the selected language
		apopUpdateContentBasedOnLanguage( selectedLanguage );

		// Fade out the language settings panel
		$( '.apop-accessibility-settings header.apop-header-top .apop-list-of-languages' ).fadeOut( 350 );

		// Updates the 'data-on' and 'data-off' attributes
		updateLabelModeSwitch();

		const getDataAccessibilityDefault = getDataAccessibilityData();
		getDataAccessibilityDefault.information.language = selectedLanguage;
		localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
	} );

	// Retrieve accessibility settings from localStorage
	const getDataLocalStorage = getDataAccessibilityData();

	// List of feature modes and their corresponding checkbox IDs
	const featureModes = {
		visionImpairedMode: '#apop-box-vision-impaired-mode',
		seizureSafeProfileMode: '#apop-box-seizure-safe-profile',
		adhdFriendlyMode: '#apop-box-adhd-friendly-mode',
		blindnessMode: '#apop-box-blindness-mode',
		epilepsySafeMode: '#apop-box-epilepsy-safe-mode',
	};

	// Function to initialize checkboxes based on saved settings
	function initializeModePresetToggle() {
		$.each( featureModes, function( key, selector ) {
			const checkbox = $( '.apop-toggle-container ' + selector );

			// Set the checkbox state based on saved feature settings
			if ( getDataLocalStorage.dynamicFeatureSet[ key ] ) {
				checkbox.prop( 'checked', true ).val( 1 );
				$( selector ).parent().parent().parent().parent().addClass( 'apop-active' );
			}
		} );
	}

	// Function to handle checkbox state changes
	function handleModePresetToggle() {
		$( '.apop-toggle-container input[type="checkbox"]' ).on( 'change', function() {
			// Find the feature key corresponding to the checkbox
			const featureKey = Object.keys( featureModes ).find( ( key ) => featureModes[ key ] === '#' + this.id );

			// Get attr by id.
			const getID = $( this ).attr( 'id' );

			if ( featureKey ) {
				const isChecked = $( this ).is( ':checked' );

				// Update the feature settings in localStorage
				const getDataAccessibilityDefault = getDataAccessibilityData();
				Object.assign( getDataAccessibilityDefault.dynamicFeatureSet, {
					visionImpairedMode: false,
					seizureSafeProfileMode: false,
					adhdFriendlyMode: false,
					blindnessMode: false,
					epilepsySafeMode: false,
					[ featureKey ]: isChecked,
				} );

				// Set all default checkboxes to false.
				const checkboxPresetToggle = [
					'#apop-box-vision-impaired-mode',
					'#apop-box-seizure-safe-profile',
					'#apop-box-adhd-friendly-mode',
					'#apop-box-blindness-mode',
					'#apop-box-epilepsy-safe-mode',
				];

				checkboxPresetToggle.forEach( ( id ) => {
					const checkbox = document.querySelector( id );
					if ( checkbox ) {
						checkbox.checked = false;
					}
				} );

				// Set the default value for this checkbox.
				$( this ).prop( 'checked', true ).val( 1 );

				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );

				// If the visionImpairedMode is toggled, trigger the appropriate function
				if ( featureKey === 'visionImpairedMode' && 'apop-box-vision-impaired-mode' === getID ) {
					apopSeizureSafeProfileMode( false );
					apopADHDFriendlyMode( false );
					apopBlindnessMode( false );
					apopEpilepsySafeMode( false );

					apopVisionImpairedMode( isChecked );
				}

				// If the seizureSafeProfileMode is toggled, trigger the appropriate function
				if ( featureKey === 'seizureSafeProfileMode' && 'apop-box-seizure-safe-profile' === getID ) {
					apopVisionImpairedMode( false );
					apopADHDFriendlyMode( false );
					apopBlindnessMode( false );
					apopEpilepsySafeMode( false );

					apopSeizureSafeProfileMode( isChecked );
				}

				// If the adhdFriendlyMode is toggled, trigger the appropriate function
				if ( featureKey === 'adhdFriendlyMode' && 'apop-box-adhd-friendly-mode' === getID ) {
					apopVisionImpairedMode( false );
					apopSeizureSafeProfileMode( false );
					apopBlindnessMode( false );
					apopEpilepsySafeMode( false );

					apopADHDFriendlyMode( isChecked );
				}

				// If the blindnessMode is toggled, trigger the appropriate function
				if ( featureKey === 'blindnessMode' && 'apop-box-blindness-mode' === getID ) {
					apopVisionImpairedMode( false );
					apopSeizureSafeProfileMode( false );
					apopADHDFriendlyMode( false );
					apopEpilepsySafeMode( false );

					apopBlindnessMode( isChecked );
				}

				// If the epilepsySafeMode is toggled, trigger the appropriate function
				if ( featureKey === 'epilepsySafeMode' && 'apop-box-epilepsy-safe-mode' === getID ) {
					apopVisionImpairedMode( false );
					apopSeizureSafeProfileMode( false );
					apopADHDFriendlyMode( false );
					apopBlindnessMode( false );

					apopEpilepsySafeMode( isChecked );
				}
			}
		} );
	}

	// Initialize checkboxes and event listeners when the window loads
	$( window ).on( 'load', function() {
		initializeModePresetToggle(); // Initialize checkboxes based on saved data
		handleModePresetToggle(); // Set up event listener for checkbox changes
	} );

	// Function to manage Vision Impaired Mode
	function apopVisionImpairedMode( status ) {
		if ( ! localStorage.getItem( apopLocalStorage ) ) {
			return;
		}

		const getDataAccessibilityDefault = getDataAccessibilityData();
		const settings = [
			{ key: 'bigger-text', prop: 'biggerText', border: true },
			{ key: 'saturation', prop: 'saturation', border: true },
			{ key: 'readable-font', prop: 'readableFont', border: false },
		];

		settings.forEach( ( { key, prop, border } ) => {
			if ( apopAjaxObject.showModules[ key ] === 'on' ) {
				getDataAccessibilityDefault[ prop ] = status;
				if ( border ) {
					getDataAccessibilityDefault.activeBorders[ prop ] = status ? 1 : 0;
				}
				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
			}
		} );

		location.reload();
	}

	// Function to manage Seizure Safe Profile Mode
	function apopSeizureSafeProfileMode( status ) {
		if ( ! localStorage.getItem( apopLocalStorage ) ) {
			return;
		}

		const getDataAccessibilityDefault = getDataAccessibilityData();
		const settings = [
			{ key: 'stop-animations', prop: 'stopAnimations', border: true },
			{ key: 'grayscale', prop: 'grayscale', border: true },
		];

		settings.forEach( ( { key, prop, border } ) => {
			if ( apopAjaxObject.showModules[ key ] === 'on' ) {
				getDataAccessibilityDefault[ prop ] = status;
				if ( border ) {
					getDataAccessibilityDefault.activeBorders[ prop ] = status ? 2 : 0;
				}
				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
			}
		} );

		location.reload();
	}

	// Function to manage ADHD Friendly Mode
	function apopADHDFriendlyMode( status ) {
		if ( ! localStorage.getItem( apopLocalStorage ) ) {
			return;
		}

		const getDataAccessibilityDefault = getDataAccessibilityData();
		const settings = [
			{ key: 'stop-animations', prop: 'stopAnimations', border: true },
			{ key: 'saturation', prop: 'saturation', border: true },
			{ key: 'reading-mask', prop: 'readingMask', border: true },
		];

		settings.forEach( ( { key, prop, border } ) => {
			if ( apopAjaxObject.showModules[ key ] === 'on' ) {
				getDataAccessibilityDefault[ prop ] = status;
				if ( border ) {
					getDataAccessibilityDefault.activeBorders[ prop ] = status ? 1 : 0;
				}
				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
			}
		} );

		location.reload();
	}

	// Function to manage Blindness Mode
	function apopBlindnessMode( status ) {
		if ( ! localStorage.getItem( apopLocalStorage ) ) {
			return;
		}

		const getDataAccessibilityDefault = getDataAccessibilityData();
		const settings = [
			{ key: 'read-page', prop: 'readPage', border: true },
			{ key: 'brightness', prop: 'brightness', border: true },
			{ key: 'contrast', prop: 'contrast', border: true },
		];

		settings.forEach( ( { key, prop, border } ) => {
			if ( apopAjaxObject.showModules[ key ] === 'on' ) {
				getDataAccessibilityDefault[ prop ] = status;
				if ( border ) {
					getDataAccessibilityDefault.activeBorders[ prop ] = status ? 3 : 0;
				}
				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
			}
		} );

		location.reload();
	}

	// Function to manage Epilepsy Safe Mode
	function apopEpilepsySafeMode( status ) {
		if ( ! localStorage.getItem( apopLocalStorage ) ) {
			return;
		}

		const getDataAccessibilityDefault = getDataAccessibilityData();
		const settings = [
			{ key: 'stop-animations', prop: 'stopAnimations', border: true },
			{ key: 'grayscale', prop: 'grayscale', border: true },
			{ key: 'mute-sounds', prop: 'muteSounds', border: true },
		];

		settings.forEach( ( { key, prop, border } ) => {
			if ( apopAjaxObject.showModules[ key ] === 'on' ) {
				getDataAccessibilityDefault[ prop ] = status;
				if ( border ) {
					getDataAccessibilityDefault.activeBorders[ prop ] = status ? 2 : 0;
				}
				localStorage.setItem( apopLocalStorage, JSON.stringify( getDataAccessibilityDefault ) );
			}
		} );

		location.reload();
	}

	// Function to update content based on the selected language
	apopUpdateContentBasedOnLanguage( getDataAccessibilityData().information.language );
	function apopUpdateContentBasedOnLanguage( language ) {
		// Define a list of valid languages
		const validLanguages = [ 'en', 'de', 'es', 'fr', 'it', 'pl', 'se', 'fi', 'pt', 'ro', 'si', 'sk', 'nl', 'dk', 'gr', 'cz', 'hu', 'lt', 'lv', 'ee', 'hr', 'ie', 'bg', 'no', 'tr', 'id', 'pt-br', 'ja', 'ko', 'zh', 'ar', 'ru', 'hi', 'uk', 'sr' ];

		// Check if the provided language is valid
		if ( validLanguages.includes( language ) ) {
			const languageData = apopAjaxObject.languages[ language ];

			// Define an array of selectors and their corresponding data keys
			const updates = [
				// Header.
				{ selector: 'nav.apop-accessibility header.apop-header-top .apop-languages .apop-text span', text: languageData.header.language },
				{ selector: 'nav.apop-accessibility header.apop-header-top .apop-site-container .apop-site-info .apop-title h2', text: languageData.header.title },
				{ selector: 'nav.apop-accessibility header.apop-header-top .apop-site-container .apop-site-info .apop-desc p span', text: languageData.header.desc },
				{ selector: 'nav.apop-accessibility header.apop-header-top .apop-site-container .apop-site-info .apop-desc p a', text: languageData.header.anchor },

				// Multi Functions Title.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-title h2', text: languageData.multiFunctionalFeature.title },

				// Vision Impaired Mode.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-vision-impaired-mode .apop-left .apop-text .apop-title span', text: languageData.multiFunctionalFeature.visionImpairedMode.title },
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-vision-impaired-mode .apop-left .apop-text .apop-desc span', text: languageData.multiFunctionalFeature.visionImpairedMode.desc },
				{ selectorOn: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-vision-impaired-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', on: languageData.multiFunctionalFeature.visionImpairedMode.on },
				{ selectorOff: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-vision-impaired-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', off: languageData.multiFunctionalFeature.visionImpairedMode.off },

				// Seizure Safe Profile.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-seizure-safe-profile-mode .apop-left .apop-text .apop-title span', text: languageData.multiFunctionalFeature.seizureSafeProfile.title },
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-seizure-safe-profile-mode .apop-left .apop-text .apop-desc span', text: languageData.multiFunctionalFeature.seizureSafeProfile.desc },
				{ selectorOn: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-seizure-safe-profile-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', on: languageData.multiFunctionalFeature.seizureSafeProfile.on },
				{ selectorOff: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-seizure-safe-profile-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', off: languageData.multiFunctionalFeature.seizureSafeProfile.off },

				// ADHD Friendly Mode.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-adhd-friendly-mode .apop-left .apop-text .apop-title span', text: languageData.multiFunctionalFeature.aDHDFriendlyMode.title },
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-adhd-friendly-mode .apop-left .apop-text .apop-desc span', text: languageData.multiFunctionalFeature.aDHDFriendlyMode.desc },
				{ selectorOn: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-adhd-friendly-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', on: languageData.multiFunctionalFeature.aDHDFriendlyMode.on },
				{ selectorOff: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-adhd-friendly-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', off: languageData.multiFunctionalFeature.aDHDFriendlyMode.off },

				// Blindness Mode.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-blindness-mode .apop-left .apop-text .apop-title span', text: languageData.multiFunctionalFeature.blindnessMode.title },
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-blindness-mode .apop-left .apop-text .apop-desc span', text: languageData.multiFunctionalFeature.blindnessMode.desc },
				{ selectorOn: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-blindness-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', on: languageData.multiFunctionalFeature.blindnessMode.on },
				{ selectorOff: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-blindness-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', off: languageData.multiFunctionalFeature.blindnessMode.off },

				// Epilepsy Safe Mode.
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-epilepsy-safe-mode .apop-left .apop-text .apop-title span', text: languageData.multiFunctionalFeature.epilepsySafeMode.title },
				{ selector: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-epilepsy-safe-mode .apop-left .apop-text .apop-desc span', text: languageData.multiFunctionalFeature.epilepsySafeMode.desc },
				{ selectorOn: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-epilepsy-safe-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', on: languageData.multiFunctionalFeature.epilepsySafeMode.on },
				{ selectorOff: 'nav.apop-accessibility .apop-container .apop-accessibility-settings .apop-multi-functional-feature .apop-box-functions .apop-box-epilepsy-safe-mode .apop-right .apop-toggle-container .label-mode-switch .label-mode-switch-inner', off: languageData.multiFunctionalFeature.epilepsySafeMode.off },

				// Content.
				{ selector: 'nav.apop-accessibility .apop-features .apop-bigger-text .apop-title h3', text: languageData.content.biggerText },
				{ selector: 'nav.apop-accessibility .apop-features .apop-cursor .apop-title h3', text: languageData.content.cursor },
				{ selector: 'nav.apop-accessibility .apop-features .apop-line-height .apop-title h3', text: languageData.content.lineHeight },
				{ selector: 'nav.apop-accessibility .apop-features .apop-letter-spacing .apop-title h3', text: languageData.content.letterSpacing },
				{ selector: 'nav.apop-accessibility .apop-features .apop-readable-font .apop-title h3', text: languageData.content.readableFont },
				{ selector: 'nav.apop-accessibility .apop-features .apop-dyslexic-font .apop-title h3', text: languageData.content.dyslexicFont },

				// Content Bottom.
				{ selector: 'nav.apop-accessibility .apop-features .apop-text-align .apop-title h3', text: languageData.contentBottom.textAlign },
				{ selector: 'nav.apop-accessibility .apop-features .apop-text-magnifier .apop-title h3', text: languageData.contentBottom.textMagnifier },
				{ selector: 'nav.apop-accessibility .apop-features .apop-highlight-links .apop-title h3', text: languageData.contentBottom.highlightLinks },

				// Colors.
				{ selector: 'nav.apop-accessibility .apop-features .apop-invert-colors .apop-title h3', text: languageData.colors.invertColors },
				{ selector: 'nav.apop-accessibility .apop-features .apop-brightness .apop-title h3', text: languageData.colors.brightness },
				{ selector: 'nav.apop-accessibility .apop-features .apop-contrast .apop-title h3', text: languageData.colors.contrast },
				{ selector: 'nav.apop-accessibility .apop-features .apop-grayscale .apop-title h3', text: languageData.colors.grayscale },
				{ selector: 'nav.apop-accessibility .apop-features .apop-saturation .apop-title h3', text: languageData.colors.saturation },

				// Orientation.
				{ selector: 'nav.apop-accessibility .apop-features .apop-reading-line .apop-title h3', text: languageData.orientation.readingLine },
				{ selector: 'nav.apop-accessibility .apop-features .apop-keyboard-navigation .apop-title h3', text: languageData.orientation.keyboardNavigation },
				{ selector: 'nav.apop-accessibility .apop-features .apop-highlight-titles .apop-title h3', text: languageData.orientation.highlightTitles },
				{ selector: 'nav.apop-accessibility .apop-features .apop-reading-mask .apop-title h3', text: languageData.orientation.readingMask },
				{ selector: 'nav.apop-accessibility .apop-features .apop-hide-images .apop-title h3', text: languageData.orientation.hideImages },
				{ selector: 'nav.apop-accessibility .apop-features .apop-highlight-all .apop-title h3', text: languageData.orientation.highlightAll },

				// Content Bottom.
				{ selector: 'nav.apop-accessibility .apop-features .apop-read-page .apop-title h3', text: languageData.orientationBottom.readPage },
				{ selector: 'nav.apop-accessibility .apop-features .apop-mute-sounds .apop-title h3', text: languageData.orientationBottom.muteSounds },
				{ selector: 'nav.apop-accessibility .apop-features .apop-stop-animations .apop-title h3', text: languageData.orientationBottom.stopAnimations },

				// Divider.
				{ selector: 'nav.apop-accessibility .apop-divider-separator .apop-content', text: languageData.divider.content },
				{ selector: 'nav.apop-accessibility .apop-divider-separator .apop-colors', text: languageData.divider.colors },
				{ selector: 'nav.apop-accessibility .apop-divider-separator .apop-orientation', text: languageData.divider.navigation },

				// ResetSettings.
				{ selector: 'nav.apop-accessibility .apop-accessibility-settings .apop-reset-settings span', text: languageData.resetSettings },

				// Footer.
				{ selector: 'nav.apop-accessibility .apop-footer-bottom .apop-icon-list-text', text: languageData.footer.accessibilityStatement },
				{ selector: 'nav.apop-accessibility footer.apop-footer-bottom .apop-divider-container .apop-divider__text', text: languageData.footer.version },
			];

			// Update each element with the corresponding text
			updates.forEach( ( update ) => {
				$( update.selector ).text( update.text );

				// Check if update.selectorOn exist.
				if ( update.selectorOn ) {
					$( update.selectorOn ).attr( 'data-on', update.on );
				}

				// Check if update.selectorOff exist.
				if ( update.selectorOff ) {
					$( update.selectorOff ).attr( 'data-off', update.off );
				}
			} );
		}
	}

	// Updates the font-size of elements except for the excluded selectors
	function apopUpdateHeadingFontSize( heading, excludedSelectors, fontSize, activeBorderValue ) {
		$( '*' ).not( excludedSelectors ).each( function() {
			// Get the current inline style of the element, or use an empty string if none exists
			let currentStyle = $( this ).attr( 'style' ) || '';

			if ( 0 === activeBorderValue ) {
				// Remove the font-size if activeBorderValue is 0
				currentStyle = currentStyle.replace( /font-size:\s*[^;]+;?/, '' );
			} else if ( [ 1, 2, 3 ].includes( activeBorderValue ) ) {
				// Check if the element has a style attribute and if it ends with a semicolon
				if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
					currentStyle += ';';
				}

				// Check if 'font-size' is already defined in the style
				if ( /font-size:\s*[^;]+;?/.test( currentStyle ) ) {
					// If it exists, replace the existing font-size with the new value
					currentStyle = currentStyle.replace( /font-size:\s*[^;]+;?/, 'font-size: ' + fontSize );
				} else {
					// If font-size is not present, append it to the style attribute
					currentStyle += ' font-size: ' + fontSize;
				}
			}

			// Trim any extra spaces and ensure there's no trailing space
			currentStyle = currentStyle.trim();

			// Set the updated style attribute back to the element
			$( this ).attr( 'style', currentStyle );
		} );
	}

	// Updates the letter-spacing of elements except for the excluded selectors
	function apopUpdateLetterSpacing( letter, excludedSelectors, letterSpacing, activeBorderValue ) {
		$( '*' ).not( excludedSelectors ).each( function() {
			// Get the current inline style of the element, or use an empty string if none exists
			let currentStyle = $( this ).attr( 'style' ) || '';

			if ( 0 === activeBorderValue ) {
				// Remove the letter-spacing if activeBorderValue is 0
				currentStyle = currentStyle.replace( /letter-spacing:\s*[^;]+;?/, '' );
			} else if ( [ 1, 2, 3 ].includes( activeBorderValue ) ) {
				// Check if the element has a style attribute and if it ends with a semicolon
				if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
					currentStyle += ';';
				}

				// Check if 'letter-spacing' is already defined in the style
				if ( /letter-spacing:\s*[^;]+;?/.test( currentStyle ) ) {
					// If it exists, replace the existing letter-spacing with the new value
					currentStyle = currentStyle.replace( /letter-spacing:\s*[^;]+;?/, 'letter-spacing: ' + letterSpacing );
				} else {
					// If letter-spacing is not present, append it to the style attribute
					currentStyle += ' letter-spacing: ' + letterSpacing;
				}
			}

			// Trim any extra spaces and ensure there's no trailing space
			currentStyle = currentStyle.trim();

			// Set the updated style attribute back to the element
			$( this ).attr( 'style', currentStyle );
		} );
	}

	// This function adjusts the text size based on the 'biggerText'
	function apopBiggerText( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'bigger-text' ] ) {
			return;
		}

		// Check if the key is 'fontSize'. If it is, the function will proceed with font size adjustments.
		if ( 'biggerText' === key ) {
			let increasePercent = 0;

			if ( 1 === activeBorderValue ) {
				increasePercent = 0.25;
			} else if ( 2 === activeBorderValue ) {
				increasePercent = 0.5;
			} else if ( 3 === activeBorderValue ) {
				increasePercent = 0.75;
			}

			// General
			const validTags = [
				'p', 'span', 'a', 'li', 'td', 'th', 'label', 'button', 'input', 'textarea',
				'strong', 'em', 'b', 'i', 'u', 'small', 'time', 'code', 'pre',
				'blockquote', 'cite', 'abbr', 'address', 'q', 'dt', 'dd',
				'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
				'mark', 'sup', 'sub', 'del', 'ins', 's',
				'kbd', 'samp', 'var',
				'legend', 'figcaption', 'summary', 'body',
			];

			$( 'body, body *' ).each( function() {
				const el = this;
				const tag = el.tagName ? el.tagName.toLowerCase() : '';

				// Skip if tag is not in validTags
				if ( ! validTags.includes( tag ) ) {
					return;
				}

				// Skip elements in apopSkipElements if needed
				if ( $( this ).is( apopSkipElements ) ) {
					return;
				}

				// Skip <li> elements that are inside another <li>.
				if ( tag === 'li' && $( this ).parents( 'li' ).length > 0 ) {
					return;
				}

				const computedStyle = window.getComputedStyle( el );

				const currentFontSize = parseFloat( computedStyle.fontSize );

				if ( ! el.dataset.originalFontSize ) {
					el.dataset.originalFontSize = currentFontSize;
				}

				const baseFontSize = parseFloat( el.dataset.originalFontSize );
				let newStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Reset to default
					newStyle = newStyle.replace( /font-size:\s*[^;]+;?/, '' );
				} else if ( baseFontSize ) {
					const newFontSize = ( baseFontSize * ( 1 + increasePercent ) ).toFixed( 2 );
					const fontSizeStr = `font-size: ${ newFontSize }px !important;`;

					if ( /font-size:\s*[^;]+;?/.test( newStyle ) ) {
						newStyle = newStyle.replace( /font-size:\s*[^;]+;?/, fontSizeStr );
					} else {
						newStyle += ( newStyle.trim().endsWith( ';' ) ? ' ' : '; ' ) + fontSizeStr;
					}
				}

				$( this ).attr( 'style', newStyle.trim() );
			} );
		}
	}

	// This function modifies the cursor size by adding and removing classes
	function apopCursor( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules.cursor ) {
			return;
		}

		// Check if the key is 'Cursor'. If it is, the function will proceed with font size adjustments.
		if ( 'cursor' === key ) {
			if ( 1 === activeBorderValue ) {
				// Add the class to the root <html> element
				$( 'html' ).addClass( 'apop-cursor-feature1' );
				$( 'html' ).removeClass( 'apop-cursor-feature2' );
				$( 'html' ).removeClass( 'apop-cursor-feature3' );
			} else if ( 2 === activeBorderValue ) {
				// Add the class to the root <html> element
				$( 'html' ).removeClass( 'apop-cursor-feature1' );
				$( 'html' ).addClass( 'apop-cursor-feature2' );
				$( 'html' ).removeClass( 'apop-cursor-feature3' );
			} else if ( 3 === activeBorderValue ) {
				// Add the class to the root <html> element
				$( 'html' ).removeClass( 'apop-cursor-feature1' );
				$( 'html' ).removeClass( 'apop-cursor-feature2' );
				$( 'html' ).addClass( 'apop-cursor-feature3' );
			} else {
				// Add the class to the root <html> element
				$( 'html' ).removeClass( 'apop-cursor-feature1' );
				$( 'html' ).removeClass( 'apop-cursor-feature2' );
				$( 'html' ).removeClass( 'apop-cursor-feature3' );
			}
		}
	}

	// This function adjusts the line height based on the 'lineHeight'
	function apopLineHeight( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'line-height' ] ) {
			return;
		}

		// Check if the key is 'lineHeight'. If it is, the function will proceed with line height adjustments.
		if ( 'lineHeight' === key ) {
			let increasePercent = 0;

			if ( 1 === activeBorderValue ) {
				increasePercent = 0.25;
			} else if ( 2 === activeBorderValue ) {
				increasePercent = 0.5;
			} else if ( 3 === activeBorderValue ) {
				increasePercent = 0.75;
			}

			// General
			const validTags = [
				'p', 'span', 'a', 'li', 'td', 'th', 'label', 'button', 'input', 'textarea',
				'strong', 'em', 'b', 'i', 'u', 'small', 'time', 'code', 'pre',
				'blockquote', 'cite', 'abbr', 'address', 'q', 'dt', 'dd',
				'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
				'mark', 'sup', 'sub', 'del', 'ins', 's',
				'kbd', 'samp', 'var',
				'legend', 'figcaption', 'summary', 'body',
			];

			$( 'body, body *' ).each( function() {
				const el = this;
				const tag = el.tagName ? el.tagName.toLowerCase() : '';

				// Skip if tag is not in validTags
				if ( ! validTags.includes( tag ) ) {
					return;
				}

				// Skip elements in apopSkipElements if needed
				if ( $( this ).is( apopSkipElements ) ) {
					return;
				}

				const computedStyle = window.getComputedStyle( el );

				// Skip <li> elements that are inside another <li>.
				if ( tag === 'li' && $( this ).parents( 'li' ).length > 0 ) {
					return;
				}

				const currentLineHeight = parseFloat( computedStyle.lineHeight );

				if ( ! el.dataset.originalLineHeight ) {
					el.dataset.originalLineHeight = currentLineHeight;
				}

				const baseLineHeight = parseFloat( el.dataset.originalLineHeight );
				let newStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Reset to default
					newStyle = newStyle.replace( /line-height:\s*[^;]+;?/, '' );
				} else if ( baseLineHeight ) {
					const newLineHeight = ( baseLineHeight * ( 1 + increasePercent ) ).toFixed( 2 );
					const lineHeightStr = `line-height: ${ newLineHeight }px !important;`;

					if ( /line-height:\s*[^;]+;?/.test( newStyle ) ) {
						newStyle = newStyle.replace( /line-height:\s*[^;]+;?/, lineHeightStr );
					} else {
						newStyle += ( newStyle.trim().endsWith( ';' ) ? ' ' : '; ' ) + lineHeightStr;
					}
				}

				$( this ).attr( 'style', newStyle.trim() );
			} );
		}
	}

	// This function adjusts the letter spacing based on the 'letterSpacing'
	function apopLetterSpacing( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'letter-spacing' ] ) {
			return;
		}

		// Check if the key is 'letterSpacing'. If it is, the function will proceed with font size adjustments.
		if ( 'letterSpacing' === key ) {
			let letterSpacing = null;

			// Determine letterSpacing value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				letterSpacing = '1px !important;';
			} else if ( 2 === activeBorderValue ) {
				letterSpacing = '3px !important;';
			} else if ( 3 === activeBorderValue ) {
				letterSpacing = '5px !important;';
			} else {
				letterSpacing = null;
			}

			// General
			$( '*' ).not( apopSkipElements ).each( function() {
				// Get the current inline style of the element, or use an empty string if none exists
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the letter-spacing if activeBorderValue is 0
					currentStyle = currentStyle.replace( /letter-spacing:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					// Check if 'letter-spacing' is already defined in the style
					if ( /letter-spacing:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing letter-spacing with the new value
						currentStyle = currentStyle.replace( /letter-spacing:\s*[^;]+;?/, 'letter-spacing: ' + letterSpacing );
					} else {
						// If letter-spacing is not present, append it to the style attribute
						currentStyle += ' letter-spacing: ' + letterSpacing;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );

			// Call the function for each heading type
			apopUpdateLetterSpacing( null, apopSkipElements, letterSpacing, activeBorderValue );
		}
	}

	// This function adjusts the readablefont based on the 'readableFont'
	function apopReadableFont( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'readable-font' ] ) {
			return;
		}

		// Check if the key is 'readableFont'. If it is, the function will proceed with font size adjustments.
		if ( 'readableFont' === key ) {
			// Update style for all elements except specific ones
			$( '*' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( ! accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( getDataAccessibilityData().dyslexicFont ) {
						currentStyle = currentStyle.replace( 'font-family: Roboto, sans-serif !important;', 'font-family: OpenDyslexic, sans-serif !important;' );
					} else {
						// Remove the font-family if accessibilityDataKey is 0
						currentStyle = currentStyle.replace( /font-family:\s*[^;]+;?/, '' );
					}
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /font-family:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing font-family with the new value
						currentStyle = currentStyle.replace( /font-family:\s*[^;]+;?/, 'font-family: Roboto, sans-serif !important;' );
					} else {
						// If font-family is not present, append it to the style attribute
						currentStyle += ' font-family: Roboto, sans-serif !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the dyslexicfont based on the 'dyslexicFont'
	function apopDyslexicFont( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'dyslexic-font' ] ) {
			return;
		}

		// Check if the key is 'dyslexicFont'. If it is, the function will proceed with font size adjustments.
		if ( 'dyslexicFont' === key && ! getDataAccessibilityData().readableFont ) {
			// Update style for all elements except specific ones
			$( '*' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( ! accessibilityDataKey ) {
					// Remove the font-family if accessibilityDataKey is 0
					currentStyle = currentStyle.replace( /font-family:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /font-family:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing font-family with the new value
						currentStyle = currentStyle.replace( /font-family:\s*[^;]+;?/, 'font-family: OpenDyslexic, sans-serif !important;' );
					} else {
						// If font-family is not present, append it to the style attribute
						currentStyle += ' font-family: OpenDyslexic, sans-serif !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the text align based on the 'textAlign'
	function apopTextAlign( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'text-align' ] ) {
			return;
		}

		// Check if the key is 'textAlign'. If it is, the function will proceed with font size adjustments.
		if ( 'textAlign' === key ) {
			let textAlign = null;

			// Determine textAlign value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				textAlign = 'left !important;';
			} else if ( 2 === activeBorderValue ) {
				textAlign = 'center !important;';
			} else if ( 3 === activeBorderValue ) {
				textAlign = 'right !important;';
			} else {
				textAlign = null;
			}

			// Update style for all elements except specific ones
			$( '*' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the text-align if activeBorderValue is 0
					currentStyle = currentStyle.replace( /text-align:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /text-align:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing text-align with the new value
						currentStyle = currentStyle.replace( /text-align:\s*[^;]+;?/, 'text-align: ' + textAlign );
					} else {
						// If text-align is not present, append it to the style attribute
						currentStyle += ' text-align: ' + textAlign;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the text magnifier based on the 'textMagnifier'
	function apopTextMagnifier( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'text-magnifier' ] ) {
			return;
		}

		// Check if the key is 'textMagnifier'. If it is, the function will proceed with font size adjustments.
		if ( 'textMagnifier' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( '.apop-markup-text-magnifier' ).hide();
				$( document ).off( 'mousemove' );
			} else if ( accessibilityDataKey ) {
				$( document ).on( 'mousemove', function( event ) {
					const windowWidth = $( window ).width();
					const windowHeight = $( window ).height();

					// Get element position
					const hoveredElement = document.elementFromPoint( event.clientX, event.clientY );

					// Skip if hovered element has class 'apop-markup-text-magnifier'
					if ( $( hoveredElement ).hasClass( 'apop-markup-text-magnifier' ) ) {
						return; // Skip further execution
					}

					// Check hovered element position
					if ( hoveredElement ) {
						const $this = $( hoveredElement ); // Convert the DOM element to a jQuery object
						const text = $this.contents().filter( function() {
							return this.nodeType === 3; // Filter only text nodes
						} ).text().trim(); // Get and trim text from the direct text nodes

						// Add a title only if the element has direct text and no child elements
						if ( text ) {
							$( '.apop-markup-text-magnifier' ).text( text );
							$( '.apop-markup-text-magnifier' ).show();
						} else {
							$( '.apop-markup-text-magnifier' ).text( '' );
							$( '.apop-markup-text-magnifier' ).hide();
						}
					}

					// Determine quadrant and calculate position
					if ( event.pageX < windowWidth / 2 && event.pageY < windowHeight / 2 ) {
						// Left Top -> Right Bottom
						$( '.apop-markup-text-magnifier' ).css( {
							left: event.pageX + 15 + 'px',
							top: event.pageY + 30 + 'px',
						} );
					} else if ( event.pageX >= windowWidth / 2 && event.pageY < windowHeight / 2 ) {
						// Right Top -> Left Bottom
						$( '.apop-markup-text-magnifier' ).css( {
							left: event.pageX - 115 + 'px',
							top: event.pageY + 30 + 'px',
						} );
					} else if ( event.pageX < windowWidth / 2 && event.pageY >= windowHeight / 2 ) {
						// Left Bottom -> Right Top
						$( '.apop-markup-text-magnifier' ).css( {
							left: event.pageX + 15 + 'px',
							top: event.pageY - 115 + 'px',
						} );
					} else {
						// Right Bottom -> Left Top
						$( '.apop-markup-text-magnifier' ).css( {
							left: event.pageX - 115 + 'px',
							top: event.pageY - 115 + 'px',
						} );
					}
				} );
			}
		}
	}

	// This function adjusts the highlight links based on the 'highlightLinks'
	function apopHighlightLinks( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'highlight-links' ] ) {
			return;
		}

		// Check if the key is 'highlightLinks'. If it is, the function will proceed with font size adjustments.
		if ( 'highlightLinks' === key ) {
			// Update style for all elements except specific ones
			$( 'a' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( ! accessibilityDataKey ) {
					// Remove the background and color if accessibilityDataKey is 0
					currentStyle = currentStyle.replace( /background:\s*[^;]+;?/, '' );
					currentStyle = currentStyle.replace( /color:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					// Handle background
					if ( /background:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing background with the new value
						currentStyle = currentStyle.replace( /background:\s*[^;]+;?/, 'background: #f7ff00 !important;' );
					} else {
						// If background is not present, append it to the style attribute
						currentStyle += ' background: #f7ff00 !important;';
					}

					// Handle color
					if ( /color:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing color with the new value
						currentStyle = currentStyle.replace( /color:\s*[^;]+;?/, 'color: #000 !important;' );
					} else {
						// If color is not present, append it to the style attribute
						currentStyle += ' color: #000 !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the invert colors based on the 'invertColors'
	function apopInvertColors( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'invert-colors' ] ) {
			return;
		}

		// Check if the key is 'invertColors'. If it is, the function will proceed with font size adjustments.
		if ( 'invertColors' === key ) {
			let invertColors = null;

			// Determine invertColors value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				invertColors = 'invert(70%) !important;';
			} else if ( 2 === activeBorderValue ) {
				invertColors = 'invert(85%) !important;';
			} else if ( 3 === activeBorderValue ) {
				invertColors = 'invert(100%) !important;';
			} else {
				invertColors = null;
			}

			// Update style for all elements except specific ones
			$( 'html, img' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /filter:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing filter with the new value
						currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, 'filter: ' + invertColors );
					} else {
						// If filter is not present, append it to the style attribute
						currentStyle += ' filter: ' + invertColors;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the brightness based on the 'brightness'
	function apopBrightness( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules.brightness ) {
			return;
		}

		// Check if the key is 'brightness'. If it is, the function will proceed with font size adjustments.
		if ( 'brightness' === key ) {
			let brightness = null;

			// Determine brightness value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				brightness = 'brightness(50%) !important;';
			} else if ( 2 === activeBorderValue ) {
				brightness = 'brightness(80%) !important;';
			} else if ( 3 === activeBorderValue ) {
				brightness = 'brightness(110%) !important;';
			} else {
				brightness = null;
			}

			// Update style for all elements except specific ones
			$( 'html' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /filter:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing filter with the new value
						currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, 'filter: ' + brightness );
					} else {
						// If filter is not present, append it to the style attribute
						currentStyle += ' filter: ' + brightness;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the contrast based on the 'contrast'
	function apopContrast( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules.contrast ) {
			return;
		}

		// Check if the key is 'contrast'. If it is, the function will proceed with font size adjustments.
		if ( 'contrast' === key ) {
			let contrast = null;

			// Determine contrast value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				contrast = 'contrast(50%) !important;';
			} else if ( 2 === activeBorderValue ) {
				contrast = 'contrast(80%) !important;';
			} else if ( 3 === activeBorderValue ) {
				contrast = 'contrast(110%) !important;';
			} else {
				contrast = null;
			}

			// Update style for all elements except specific ones
			$( 'html' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /filter:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing filter with the new value
						currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, 'filter: ' + contrast );
					} else {
						// If filter is not present, append it to the style attribute
						currentStyle += ' filter: ' + contrast;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the grayscale based on the 'grayscale'
	function apopGrayscale( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules.grayscale ) {
			return;
		}

		// Check if the key is 'grayscale'. If it is, the function will proceed with font size adjustments.
		if ( 'grayscale' === key ) {
			let grayscale = null;

			// Determine grayscale value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				grayscale = 'grayscale(33%) !important;';
			} else if ( 2 === activeBorderValue ) {
				grayscale = 'grayscale(66%) !important;';
			} else if ( 3 === activeBorderValue ) {
				grayscale = 'grayscale(100%) !important;';
			} else {
				grayscale = null;
			}

			// Update style for all elements except specific ones
			$( 'html' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /filter:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing filter with the new value
						currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, 'filter: ' + grayscale );
					} else {
						// If filter is not present, append it to the style attribute
						currentStyle += ' filter: ' + grayscale;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the saturation based on the 'saturation'
	function apopSaturation( key, activeBorderValue ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules.saturation ) {
			return;
		}

		// Check if the key is 'grayscale'. If it is, the function will proceed with font size adjustments.
		if ( 'saturation' === key ) {
			let saturation = null;

			// Determine saturation value based on activeBorderValue
			if ( 1 === activeBorderValue ) {
				saturation = 'saturate(150%) !important;';
			} else if ( 2 === activeBorderValue ) {
				saturation = 'saturate(200%) !important;';
			} else if ( 3 === activeBorderValue ) {
				saturation = 'saturate(250%) !important;';
			} else {
				saturation = null;
			}

			// Update style for all elements except specific ones
			$( 'html' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';

				if ( 0 === activeBorderValue ) {
					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );
				} else if ( 1 === activeBorderValue || 2 === activeBorderValue || 3 === activeBorderValue ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /filter:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing filter with the new value
						currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, 'filter: ' + saturation );
					} else {
						// If filter is not present, append it to the style attribute
						currentStyle += ' filter: ' + saturation;
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the reading line based on the 'readingLine'
	function apopReadingLine( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'reading-line' ] ) {
			return;
		}

		// Check if the key is 'readingLine'. If it is, the function will proceed with font size adjustments.
		if ( 'readingLine' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( '.apop-markup-reading-line' ).removeClass( 'apop-active' );
			} else if ( accessibilityDataKey ) {
				$( '.apop-markup-reading-line' ).addClass( 'apop-active' );
				$( document ).mousemove( function( event ) {
					// Get the X and Y coordinates of the mouse
					const mouseY = event.pageY; // Vertical position

					// Apply the Y position to the 'top' style of the '.apop-markup-reading-line' element
					$( '.apop-markup-reading-line' ).css( 'top', mouseY + 'px' );
				} );
			}
		}
	}

	// This function adjusts the keyboard navigation based on the 'keyboardNavigation'
	function apopKeyboardNavigation( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'keyboard-navigation' ] ) {
			return;
		}

		// Ensure hotkeys library is loaded.
		if ( typeof hotkeys === 'undefined' ) {
			console.error( 'Hotkeys library is not loaded.' );
			return;
		}

		/**
		 * Initialize hotkeys.
		 * @param {Object} options - Configurable hotkey options.
		 */
		function hotKeys( options ) {
			/** Open popup by hotkey. */
			openInterface( options );

			/** Navigate to next/prev Menu. */
			focusElements( options.hotKeyMenu, 'nav, [role="navigation"]' );

			/** Navigate to next/prev Heading. */
			focusElements( options.hotKeyHeadings, 'h1, h2, h3, h4, h5, h6, [role="heading"]' );

			/** Navigate to next/prev Form. */
			focusElements( options.hotKeyForms, 'form:not([disabled])' );

			/** Navigate to next/prev Button. */
			focusElements( options.hotKeyButtons, 'button:not([disabled]), [role="button"]:not([disabled])' );

			/** Navigate to next/prev Graphic. */
			focusElements( options.hotKeyGraphics, 'img, picture, svg' );

			/**
			 * Enable/Disable controls by pressing Spacebar.
			 * @param {KeyboardEvent} e - The keyboard event object.
			 */
			document.body.onkeydown = function( e ) {
				const keyPressed = e.keyCode || e.charCode || e.which;

				/** Spacebar pressed. */
				if ( keyPressed === 32 ) {
					spacePressed( e );
				}
			};
		}

		/**
		 * Set focus on next/prev elements.
		 * @param {string} shortcutKey - Key for triggering focus.
		 * @param {string} selector    - Selector for target elements.
		 */
		function focusElements( shortcutKey, selector ) {
			// Register hotkeys for both forward and backward navigation
			hotkeys( shortcutKey + ', shift+' + shortcutKey, function( event, handler ) {
				// Check if Keyboard Navigation mode is active
				if ( ! document.body.classList.contains( 'apop-keyboard-navigation' ) ) {
					console.warn( 'Keyboard Navigation is not active.' );
					return;
				}

				// Prevent the default browser behavior
				event.preventDefault();

				// Select all elements based on the provided selector
				let elements = document.querySelectorAll( selector );

				// Convert the NodeList to an array to allow filtering
				elements = Array.from( elements ).filter( ( element ) => {
				// Exclude elements that have the 'apop-heading' class
					return ! element.classList.contains( 'apop-heading' );
				} );

				// Iterate through all selected elements
				elements.forEach( ( element ) => {
				// Check if the element has the 'apop-heading' class
					if ( element.classList.contains( 'apop-heading' ) ) {
					// Remove the element from the DOM if it has the 'apop-heading' class
						element.remove();
					}
				} );

				if ( ! elements.length ) {
					console.warn( `No elements found for selector: ${ selector }` );
					return;
				}

				// Determine navigation direction
				const forward = ! handler.key.startsWith( 'shift+' );

				// Get the currently focused element
				const currentIndex = Array.from( elements ).findIndex( ( el ) => el === document.activeElement );

				// Calculate the next index
				const nextIndex = forward
					? ( currentIndex + 1 ) % elements.length
					: ( currentIndex - 1 + elements.length ) % elements.length;

				// Set focus on the next element
				const nextElement = elements[ nextIndex ];
				nextElement.setAttribute( 'tabindex', '-1' ); // Ensure element is focusable
				nextElement.focus();

				// console.log( `Focused element index: ${ nextIndex }` );
			} );
		}

		/**
		 * Placeholder function for opening a popup interface.
		 * @param {Object} options - Configurable options for popup behavior.
		 */
		function openInterface( options ) {
			/* eslint no-unused-vars: "off" */
			// console.log( 'Open interface triggered with options:', options );
		}

		/**
		 * Handle Spacebar pressed for enabling/disabling controls.
		 * @param {Event} event - The keydown event.
		 */
		function spacePressed( event ) {
			event.preventDefault();
			// console.log( 'Spacebar pressed. Toggle controls here.' );
			// Implement your logic for enabling/disabling controls.
		}

		// Check if the key is 'highlightTitles'. If it is, the function will proceed with font size adjustments.
		if ( 'keyboardNavigation' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( 'body' ).removeClass( 'apop-keyboard-navigation' );
			} else if ( accessibilityDataKey ) {
				$( 'body' ).addClass( 'apop-keyboard-navigation' );
				if ( typeof accessibilityHotkeys !== 'undefined' ) {
					hotKeys( accessibilityHotkeys );
				} else {
					console.error( 'accessibilityHotkeys object is undefined.' );
				}
			}
		}
	}

	// This function adjusts the highlight titles based on the 'highlightTitles'
	function apopHighlightTitles( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'highlight-titles' ] ) {
			return;
		}

		// Check if the key is 'highlightTitles'. If it is, the function will proceed with font size adjustments.
		if ( 'highlightTitles' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( 'body' ).removeClass( 'apop-highlight-titles' );
			} else if ( accessibilityDataKey ) {
				$( 'body' ).addClass( 'apop-highlight-titles' );
			}
		}
	}

	// This function adjusts the reading mask based on the 'readingmask'
	function apopReadingMask( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'reading-mask' ] ) {
			return;
		}

		// Check if the key is 'readingmask'. If it is, the function will proceed with font size adjustments.
		if ( 'readingMask' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( '.apop-markup-reading-mask' ).removeClass( 'apop-active' );
			} else if ( accessibilityDataKey ) {
				$( '.apop-markup-reading-mask' ).addClass( 'apop-active' );
				$( document ).mousemove( function( event ) {
					// Get the vertical position of the cursor within the viewport
					const cursorYPosition = event.clientY;

					// Define the height of the focus box (the area that remains visible)
					const focusHeight = 200;
					const halfFocusHeight = focusHeight / 2;

					// Calculate the top mask height, subtracting half of the focus area height
					const topMaskHeight = cursorYPosition - halfFocusHeight;

					// Calculate the bottom mask height based on the remaining space after the focus area
					const bottomMaskHeight = $( window ).height() - cursorYPosition - halfFocusHeight;

					// Apply the calculated height to the top mask
					$( '.apop-markup-reading-mask.apop-top' ).css( 'height', topMaskHeight + 'px' );

					// Apply the calculated height to the bottom mask
					$( '.apop-markup-reading-mask.apop-bottom' ).css( 'height', bottomMaskHeight + 'px' );
				} );
			}
		}
	}

	// This function adjusts the hide images based on the 'hideImages'
	function apopHideImages( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'hide-images' ] ) {
			return;
		}

		// Check if the key is 'hideImages'. If it is, the function will proceed with font size adjustments.
		if ( 'hideImages' === key ) {
			// Update style for all elements except specific ones
			$( 'img' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';
				if ( ! accessibilityDataKey ) {
					// Remove the visibility if accessibilityDataKey is 0
					currentStyle = currentStyle.replace( /visibility:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /visibility:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing visibility with the new value
						currentStyle = currentStyle.replace( /visibility:\s*[^;]+;?/, 'visibility: hidden !important;' );
					} else {
						// If visibility is not present, append it to the style attribute
						currentStyle += ' visibility: hidden !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );

			// Update style for all elements except specific ones
			$( '*' ).not( apopSkipElements ).each( function() {
				let currentStyle = $( this ).attr( 'style' ) || '';
				if ( ! accessibilityDataKey ) {
					// Remove the background-size if accessibilityDataKey is 0
					currentStyle = currentStyle.replace( /background-size:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle.trim() && ! /;$/.test( currentStyle.trim() ) ) {
						currentStyle += ';';
					}

					if ( /background-size:\s*[^;]+;?/.test( currentStyle ) ) {
						// If it exists, replace the existing background-size with the new value
						currentStyle = currentStyle.replace( /background-size:\s*[^;]+;?/, 'background-size: 0 0 !important;' );
					} else {
						// If background-size is not present, append it to the style attribute
						currentStyle += ' background-size: 0 0 !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle = currentStyle.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle );
			} );
		}
	}

	// This function adjusts the highlight all based on the 'highlightAll'
	function apopHighlightAll( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'highlight-all' ] ) {
			return;
		}

		// Check if the key is 'highlightAll'. If it is, the function will proceed with font size adjustments.
		if ( 'highlightAll' === key ) {
			// Update style for all elements except specific ones
			if ( ! accessibilityDataKey ) {
				$( 'body' ).removeClass( 'apop-highlight-all' );
			} else if ( accessibilityDataKey ) {
				$( 'body' ).addClass( 'apop-highlight-all' );
			}
		}
	}

	// This function adjusts the read page based on the 'readPage'
	function apopReadPage( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'read-page' ] ) {
			return;
		}

		// Check if the key is 'readPage'. If it is, the function will proceed with font size adjustments.
		if ( 'readPage' === key ) {
			// Update style for all elements except specific ones
			if ( accessibilityDataKey ) {
				let currentlySpeakingElement = null; // To track the currently highlighted element

				// Event delegation to capture clicks on any element except some form elements
				$( document ).on( 'click', '*', function( event ) {
					// Prevent bubbling to avoid selecting the whole page's text
					event.stopPropagation();

					if ( ! getDataAccessibilityData()[ key ] ) {
						return;
					}

					// Ignore input, textarea, and select elements
					if ( $( this ).is( 'input, textarea, select, .apop-plugin-onetap, .apop-plugin-onetap *, .apop-toggle, .apop-toggle *' ) ) {
						return;
					}

					// Get only the direct text from the clicked element
					const textToSpeak = $( this ).clone() // Clone the element to manipulate
						.children() // Remove the children elements
						.remove() // Remove child elements
						.end() // Go back to the original element
						.text().trim(); // Get the text and trim any spaces

					// Check if the text is not empty
					if ( textToSpeak.length > 0 ) {
						// Stop any ongoing speech before continuing
						window.speechSynthesis.cancel();

						// Move the apop-highlight to the new element
						if ( currentlySpeakingElement ) {
							$( currentlySpeakingElement ).removeClass( 'apop-highlight' ); // Remove apop-highlight from the previous element
						}
						$( this ).addClass( 'apop-highlight' ); // Add apop-highlight to the newly clicked element
						currentlySpeakingElement = this; // Store the currently highlighted element

						// Use Web Speech API to convert the text to speech
						const speech = new SpeechSynthesisUtterance( textToSpeak );

						// Remove apop-highlight when speech ends
						speech.onend = function() {
							$( currentlySpeakingElement ).removeClass( 'apop-highlight' );
							currentlySpeakingElement = null; // Reset the highlighted element
						};

						window.speechSynthesis.speak( speech );
					}

					// Prevent the default action if the clicked element is a link
					// if ( $( this ).is( 'a' ) ) {
					// 	event.preventDefault();
					// }
				} );
			}
		}
	}

	// This function adjusts the mute sounds based on the 'muteSounds'
	function apopMuteSounds( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'mute-sounds' ] ) {
			return;
		}

		// Check if the key is 'muteSounds'.
		if ( 'muteSounds' === key ) {
			if ( ! accessibilityDataKey ) {
				// Unmute all video and audio
				$( 'audio, video' ).not( apopSkipElements ).each( function() {
					$( this ).prop( 'muted', false );
				} );

				// Unmute all YouTube iframes
				$( 'iframe[src*="youtube.com"]' ).each( function() {
					const src = $( this ).attr( 'src' );
					if ( src.includes( 'mute=1' ) ) {
						$( this ).attr( 'src', src.replace( 'mute=1', '' ) );
					}
				} );
			} else if ( accessibilityDataKey ) {
				// Mute all video and audio
				$( 'audio, video' ).not( apopSkipElements ).each( function() {
					$( this ).prop( 'muted', true );
				} );

				// Mute all YouTube iframes
				$( 'iframe[src*="youtube.com"]' ).each( function() {
					const src = $( this ).attr( 'src' );
					if ( ! src.includes( 'mute=1' ) ) {
						$( this ).attr( 'src', src + ( src.includes( '?' ) ? '&' : '?' ) + 'mute=1' );
					}
				} );
			}
		}
	}

	// This function adjusts the stop animations based on the 'stopAnimations'
	function apopStopAnimations( key, accessibilityDataKey ) {
		// if value off, return.
		if ( 'off' === apopAjaxObject.showModules[ 'stop-animations' ] ) {
			return;
		}

		// Check if the key is 'stopAnimations'. If it is, the function will proceed with font size adjustments.
		if ( 'stopAnimations' === key ) {
			// Update style for all elements except specific ones
			$( '*' ).not( apopSkipElements ).each( function() {
				// Transition.
				let currentStyle1 = $( this ).attr( 'style' ) || '';
				if ( ! accessibilityDataKey ) {
					// Remove the background and color if accessibilityDataKey is 0
					currentStyle1 = currentStyle1.replace( /transition:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle1.trim() && ! /;$/.test( currentStyle1.trim() ) ) {
						currentStyle1 += ';';
					}

					if ( /transition:\s*[^;]+;?/.test( currentStyle1 ) ) {
						// If it exists, replace the existing transition with the new value
						currentStyle1 = currentStyle1.replace( /transition:\s*[^;]+;?/, 'transition: none !important;' );
					} else {
						// If transition is not present, append it to the style attribute
						currentStyle1 += ' transition: none !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle1 = currentStyle1.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle1 );

				// Animations.
				let currentStyle2 = $( this ).attr( 'style' ) || '';
				if ( ! accessibilityDataKey ) {
					// Remove the background and color if accessibilityDataKey is 0
					currentStyle2 = currentStyle2.replace( /animation:\s*[^;]+;?/, '' );
				} else if ( accessibilityDataKey ) {
					// Check if the element has a style attribute and if it ends with a semicolon
					if ( currentStyle2.trim() && ! /;$/.test( currentStyle2.trim() ) ) {
						currentStyle2 += ';';
					}

					if ( /animation:\s*[^;]+;?/.test( currentStyle2 ) ) {
						// If it exists, replace the existing animation with the new value
						currentStyle2 = currentStyle2.replace( /animation:\s*[^;]+;?/, 'animation: none !important;' );
					} else {
						// If animation is not present, append it to the style attribute
						currentStyle2 += ' animation: none !important;';
					}
				}

				// Trim any extra spaces and ensure there's no trailing space
				currentStyle2 = currentStyle2.trim();

				// Set the updated style attribute back to the element
				$( this ).attr( 'style', currentStyle2 );
			} );
		}
	}

	// List of apopGetTlements and their keys
	const apopGetTlements = [
		{ selector: '.apop-bigger-text', key: 'biggerText' },
		{ selector: '.apop-cursor', key: 'cursor' },
		{ selector: '.apop-line-height', key: 'lineHeight' },
		{ selector: '.apop-letter-spacing', key: 'letterSpacing' },
		{ selector: '.apop-readable-font', key: 'readableFont' },
		{ selector: '.apop-dyslexic-font', key: 'dyslexicFont' },
		{ selector: '.apop-text-align', key: 'textAlign' },
		{ selector: '.apop-text-magnifier', key: 'textMagnifier' },
		{ selector: '.apop-highlight-links', key: 'highlightLinks' },
		{ selector: '.apop-invert-colors', key: 'invertColors' },
		{ selector: '.apop-brightness', key: 'brightness' },
		{ selector: '.apop-contrast', key: 'contrast' },
		{ selector: '.apop-grayscale', key: 'grayscale' },
		{ selector: '.apop-saturation', key: 'saturation' },
		{ selector: '.apop-reading-line', key: 'readingLine' },
		{ selector: '.apop-keyboard-navigation', key: 'keyboardNavigation' },
		{ selector: '.apop-highlight-titles', key: 'highlightTitles' },
		{ selector: '.apop-reading-mask', key: 'readingMask' },
		{ selector: '.apop-hide-images', key: 'hideImages' },
		{ selector: '.apop-highlight-all', key: 'highlightAll' },
		{ selector: '.apop-read-page', key: 'readPage' },
		{ selector: '.apop-mute-sounds', key: 'muteSounds' },
		{ selector: '.apop-stop-animations', key: 'stopAnimations' },
	];

	// Utility function to update class based on current value
	function apopToggleLevelClass( $element, currentValue ) {
		const levels = [ 'apop-lv1', 'apop-lv2', 'apop-lv3' ];
		$element.removeClass( levels.join( ' ' ) );

		if ( currentValue >= 1 && currentValue <= 3 ) {
			$element.addClass( levels[ currentValue - 1 ] );
		}
	}

	// Toggles the 'apop-active' class on the provided element
	function toggleActiveClass( $element ) {
		$element.toggleClass( 'apop-active ' );
	}

	// Utility function to handle click events
	let activeStagedValue = 0;
	function apopHandleClick( $element, key, accessibilityData, useActiveBorder ) {
		$element.on( 'click', function() {
			accessibilityData = getDataAccessibilityData();
			if ( useActiveBorder ) {
				activeStagedValue = accessibilityData.activeBorders[ key ] = ( accessibilityData.activeBorders[ key ] + 1 ) % 4;
				accessibilityData[ key ] = activeStagedValue !== 0;

				apopToggleLevelClass( $element, activeStagedValue );
				apopBiggerText( key, activeStagedValue );
				apopCursor( key, activeStagedValue );
				apopLineHeight( key, activeStagedValue );
				apopLetterSpacing( key, activeStagedValue );
				apopTextAlign( key, activeStagedValue );
				apopInvertColors( key, activeStagedValue );
				apopBrightness( key, activeStagedValue );
				apopContrast( key, activeStagedValue );
				apopGrayscale( key, activeStagedValue );
				apopSaturation( key, activeStagedValue );
			} else {
				accessibilityData[ key ] = ! accessibilityData[ key ];
				toggleActiveClass( $element, accessibilityData[ key ] );
				apopReadableFont( key, accessibilityData[ key ] );
				apopDyslexicFont( key, accessibilityData[ key ] );
				apopTextMagnifier( key, accessibilityData[ key ] );
				apopHighlightLinks( key, accessibilityData[ key ] );
				apopReadingLine( key, accessibilityData[ key ] );
				apopKeyboardNavigation( key, accessibilityData[ key ] );
				apopHighlightTitles( key, accessibilityData[ key ] );
				apopReadingMask( key, accessibilityData[ key ] );
				apopHideImages( key, accessibilityData[ key ] );
				apopHighlightAll( key, accessibilityData[ key ] );
				apopReadPage( key, accessibilityData[ key ] );
				apopMuteSounds( key, accessibilityData[ key ] );
				apopStopAnimations( key, accessibilityData[ key ] );
			}

			localStorage.setItem( apopLocalStorage, JSON.stringify( accessibilityData ) );
		} );
	}

	// Initialize functionality for multiple apopGetTlements
	function apopInitAccessibilityHandlers( accessibilityData ) {
		apopGetTlements.forEach( ( { selector, key } ) => {
			const $element = $( `nav.apop-accessibility.apop-plugin-onetap .apop-accessibility-settings ${ selector }` );
			if ( $element.length ) {
				// Use activeBorder for some keys, otherwise, just toggle true/false
				const useActiveBorder = ! [
					'readableFont',
					'dyslexicFont',
					'textMagnifier',
					'highlightLinks',
					'readingLine',
					'keyboardNavigation',
					'highlightTitles',
					'readingMask',
					'hideImages',
					'highlightAll',
					'readPage',
					'muteSounds',
					'stopAnimations',
				].includes( key );

				apopHandleClick( $element, key, accessibilityData, useActiveBorder );
			}
		} );
	}

	// Handles the application of accessibility features on elements based on user settings
	function handleAccessibilityFeatures() {
		apopGetTlements.forEach( ( { selector, key } ) => {
			const $element = $( `nav.apop-accessibility.apop-plugin-onetap .apop-accessibility-settings ${ selector }` );
			if ( $element.length && getDataAccessibilityData()[ key ] !== undefined ) {
				const useActiveBorder = ! [
					'readableFont',
					'dyslexicFont',
					'textMagnifier',
					'highlightLinks',
					'readingLine',
					'keyboardNavigation',
					'highlightTitles',
					'readingMask',
					'hideImages',
					'highlightAll',
					'readPage',
					'muteSounds',
					'stopAnimations',
				].includes( key );

				if ( useActiveBorder ) {
					if ( getDataAccessibilityData().activeBorders[ key ] !== undefined ) {
						apopToggleLevelClass( $element, getDataAccessibilityData().activeBorders[ key ] );
						if ( 0 !== getDataAccessibilityData().activeBorders[ key ] ) {
							apopBiggerText( key, getDataAccessibilityData().activeBorders[ key ] );
							apopCursor( key, getDataAccessibilityData().activeBorders[ key ] );
							apopLineHeight( key, getDataAccessibilityData().activeBorders[ key ] );
							apopLetterSpacing( key, getDataAccessibilityData().activeBorders[ key ] );
							apopInvertColors( key, getDataAccessibilityData().activeBorders[ key ] );
							apopBrightness( key, getDataAccessibilityData().activeBorders[ key ] );
							apopContrast( key, getDataAccessibilityData().activeBorders[ key ] );
							apopGrayscale( key, getDataAccessibilityData().activeBorders[ key ] );
							apopSaturation( key, getDataAccessibilityData().activeBorders[ key ] );
							apopTextAlign( key, getDataAccessibilityData().activeBorders[ key ] );
						}
					}
				} else if ( getDataAccessibilityData()[ key ] !== undefined ) {
					if ( getDataAccessibilityData()[ key ] !== undefined && getDataAccessibilityData()[ key ] ) {
						toggleActiveClass( $element, getDataAccessibilityData()[ key ] );
						apopHideImages( key, getDataAccessibilityData()[ key ] );
						apopReadableFont( key, getDataAccessibilityData()[ key ] );
						apopDyslexicFont( key, getDataAccessibilityData()[ key ] );
						apopHighlightTitles( key, getDataAccessibilityData()[ key ] );
						apopHighlightAll( key, getDataAccessibilityData()[ key ] );
						apopReadingLine( key, getDataAccessibilityData()[ key ] );
						apopTextMagnifier( key, getDataAccessibilityData()[ key ] );
						apopHighlightLinks( key, getDataAccessibilityData()[ key ] );
						apopStopAnimations( key, getDataAccessibilityData()[ key ] );
						apopReadPage( key, getDataAccessibilityData()[ key ] );
						apopReadingMask( key, getDataAccessibilityData()[ key ] );
						apopKeyboardNavigation( key, getDataAccessibilityData()[ key ] );
						apopMuteSounds( key, getDataAccessibilityData()[ key ] );
					}
				}
			}
		} );

		// Initialize handlers
		apopInitAccessibilityHandlers( getDataAccessibilityData() );
	}
	handleAccessibilityFeatures();

	// Updates the 'data-on' and 'data-off' attributes
	updateLabelModeSwitch();

	// Reset settings
	$( document ).on( 'click', 'nav.apop-accessibility.apop-plugin-onetap .apop-reset-settings span', function( event ) {
		event.stopPropagation(); // Ensure this doesn't trigger auto-close

		// Select all elements with the class .apop-box-feature
		$( 'nav.apop-accessibility.apop-plugin-onetap .apop-box-feature' ).each( function() {
			// Remove specified classes
			$( this ).removeClass( 'apop-lv1 apop-lv2 apop-lv3 apop-active' );
		} );

		// Check if the localStorage item exists
		if ( localStorage.getItem( apopLocalStorage ) ) {
			// Parse the existing localStorage item
			const currentSettings = JSON.parse( localStorage.getItem( apopLocalStorage ) );

			// Check if any of the specified values are true
			const hasActiveSettings =
				currentSettings.dynamicFeatureSet.visionImpairedMode ||
				currentSettings.dynamicFeatureSet.seizureSafeProfileMode ||
				currentSettings.dynamicFeatureSet.adhdFriendlyMode ||
				currentSettings.dynamicFeatureSet.blindnessMode ||
				currentSettings.dynamicFeatureSet.epilepsySafeMode ||
				currentSettings.biggerText ||
				currentSettings.cursor ||
				currentSettings.lineHeight ||
				currentSettings.letterSpacing ||
				currentSettings.readableFont ||
				currentSettings.dyslexicFont ||
				currentSettings.textAlign ||
				currentSettings.textMagnifier ||
				currentSettings.highlightLinks ||
				currentSettings.invertColors ||
				currentSettings.brightness ||
				currentSettings.contrast ||
				currentSettings.grayscale ||
				currentSettings.saturation ||
				currentSettings.readingLine ||
				currentSettings.keyboardNavigation ||
				currentSettings.highlightTitles ||
				currentSettings.readingMask ||
				currentSettings.hideImages ||
				currentSettings.highlightAll ||
				currentSettings.readPage ||
				currentSettings.muteSounds ||
				currentSettings.stopAnimations ||
				currentSettings.information.language;

			if ( currentSettings.textMagnifier ) {
				$( document ).off( 'mousemove' );
			}

			if ( hasActiveSettings ) {
				// Remove the 'apop-active' class from all country flag images
				$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img' ).removeClass( 'apop-active' );

				// Add the 'apop-active' class to the image with the alt attribute matching the selected language
				$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img[alt="en"]' ).addClass( 'apop-active' );

				// Remove the 'apop-active' class from all country flag images
				$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img' ).removeClass( 'apop-active' );

				// Add the 'apop-active' class to the image with the alt attribute matching the selected language
				$( 'nav.apop-accessibility .apop-accessibility-settings .apop-languages .apop-icon img[alt="' + apopAjaxObject.getSettings.language + '"]' ).addClass( 'apop-active' );

				// Reset language
				apopUpdateContentBasedOnLanguage( apopAjaxObject.getSettings.language );

				// Remove localStorage item if any value is true
				localStorage.removeItem( apopLocalStorage );

				// Create a new localStorage item with default values
				localStorage.setItem( apopLocalStorage, JSON.stringify( apopProAccessibilityDefault ) );

				// Reset Mode Preset Toggle.
				const checkboxPresetToggle = [
					'#apop-box-vision-impaired-mode',
					'#apop-box-seizure-safe-profile',
					'#apop-box-adhd-friendly-mode',
					'#apop-box-blindness-mode',
					'#apop-box-epilepsy-safe-mode',
				];

				checkboxPresetToggle.forEach( ( id ) => {
					const checkbox = document.querySelector( id );
					if ( checkbox ) {
						checkbox.checked = false;
					}
				} );

				// Remove the active class for the currently active preset mode.
				$( 'nav.apop-accessibility.apop-plugin-onetap .apop-functional-feature' ).removeClass( 'apop-active' );

				// Remove style inline
				$( '*' ).not( apopSkipElements ).each( function() {
					let currentStyle = $( this ).attr( 'style' ) || '';

					// Reset (Bigger Text)
					currentStyle = currentStyle.replace( /font-size:\s*[^;]+;?/, '' );

					// Reset (Line Height)
					currentStyle = currentStyle.replace( /line-height:\s*[^;]+;?/, '' );

					// Reset (Letter Spacing)
					currentStyle = currentStyle.replace( /letter-spacing:\s*[^;]+;?/, '' );

					// Reset (Text Align)
					currentStyle = currentStyle.replace( /text-align:\s*[^;]+;?/, '' );

					// Reset (Readable Font & Dyslexic Font)
					currentStyle = currentStyle.replace( /font-family:\s*[^;]+;?/, '' );

					// Reset (Hide Images)
					currentStyle = currentStyle.replace( /background-size:\s*[^;]+;?/, '' );
					currentStyle = currentStyle.replace( /visibility:\s*[^;]+;?/, '' );

					// Reset (Stop Animations)
					currentStyle = currentStyle.replace( /transition:\s*[^;]+;?/, '' );
					currentStyle = currentStyle.replace( /animation:\s*[^;]+;?/, '' );

					// Trim any extra spaces and ensure there's no trailing space
					currentStyle = currentStyle.trim();

					// Set the updated style attribute back to the element
					$( this ).attr( 'style', currentStyle );
				} );

				// Remove style inline
				$( 'a' ).not( apopSkipElements ).each( function() {
					let currentStyle = $( this ).attr( 'style' ) || '';

					// Reset (Highlight Links)
					currentStyle = currentStyle.replace( /background:\s*[^;]+;?/, '' );
					currentStyle = currentStyle.replace( /color:\s*[^;]+;?/, '' );

					// Trim any extra spaces and ensure there's no trailing space
					currentStyle = currentStyle.trim();

					// Set the updated style attribute back to the element
					$( this ).attr( 'style', currentStyle );
				} );

				// Remove style inline
				$( 'img' ).not( apopSkipElements ).each( function() {
					let currentStyle = $( this ).attr( 'style' ) || '';

					// Reset (Hide Images)
					currentStyle = currentStyle.replace( /visibility:\s*[^;]+;?/, '' );

					// Trim any extra spaces and ensure there's no trailing space
					currentStyle = currentStyle.trim();

					// Set the updated style attribute back to the element
					$( this ).attr( 'style', currentStyle );
				} );

				// ============= Content =============

				// Reset (Cursor)
				$( 'html' ).removeClass( 'apop-cursor-feature1' );
				$( 'html' ).removeClass( 'apop-cursor-feature2' );
				$( 'html' ).removeClass( 'apop-cursor-feature3' );

				// Reset (Highlight titles)
				$( 'body' ).removeClass( 'apop-highlight-titles' );

				// Reset (Highlight all)
				$( 'body' ).removeClass( 'apop-highlight-all' );

				// ============= Content Bottom =============

				// Reset (Text magnifier)
				$( '.apop-markup-text-magnifier' ).hide();

				// ============= Colors =============

				$( 'html, img' ).not( apopSkipElements ).each( function() {
					let currentStyle = $( this ).attr( 'style' ) || '';

					// Remove the filter if activeBorderValue is 0
					currentStyle = currentStyle.replace( /filter:\s*[^;]+;?/, '' );

					// Trim any extra spaces and ensure there's no trailing space
					currentStyle = currentStyle.trim();

					// Set the updated style attribute back to the element
					$( this ).attr( 'style', currentStyle );
				} );

				// ============= Orientation =============

				// Reset (Reading line)
				$( '.apop-markup-reading-line' ).removeClass( 'apop-active' );

				// Reset (Keyboard navigation)
				$( 'body' ).removeClass( 'apop-keyboard-navigation' );

				// Reset (Reading mask)
				$( '.apop-markup-reading-mask' ).removeClass( 'apop-active' );

				// ============= Orientation bottom =============

				// Unmute all video and audio
				$( 'audio, video' ).not( apopSkipElements ).each( function() {
					$( this ).prop( 'muted', false );
				} );

				// Unmute all YouTube iframes
				$( 'iframe[src*="youtube.com"]' ).each( function() {
					const src = $( this ).attr( 'src' );
					if ( src.includes( 'mute=1' ) ) {
						$( this ).attr( 'src', src.replace( 'mute=1', '' ) );
					}
				} );
			}
		} else {
			// Create localStorage item if it does not exist
			localStorage.setItem( apopLocalStorage, JSON.stringify( apopProAccessibilityDefault ) );
		}
	} );
}( jQuery ) );
