<?php
/**
 * Accessibility Settings Option for Onetap.
 *
 * @link       https://wponetap.com
 * @since      1.0.0
 *
 * @package    Accessibility_Plugin_Onetap_Pro_Options
 * @subpackage Accessibility_Plugin_Onetap_Pro_Options/admin
 */

/**
 * Accessibility Settings Option for Onetap.
 *
 * Handles the settings related to accessibility in the Onetap Pro plugin.
 *
 * @package    Accessibility_Plugin_Onetap_Pro_Options
 * @subpackage Accessibility_Plugin_Onetap_Pro_Options/admin
 * @author     OneTap <support@wponetap.com>
 */
class Onetap_Pro_Accessibility_Settings_Options {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Settings api.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $settings_api    The options of this plugin.
	 */
	private $settings_api;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string $plugin_name      The name of this plugin.
	 * @param      string $version          The version of this plugin.
	 * @param      object $settings_manager The settings manager of this plugin.
	 */
	public function __construct( $plugin_name, $version, $settings_manager ) {

		$this->plugin_name  = $plugin_name;
		$this->version      = $version;
		$this->settings_api = $settings_manager;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_enqueue_style( $this->plugin_name . '-admin', ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/css/apop-admin-menu.min.css', array(), $this->version, 'all' );
	}

	/**
	 * Returns the URL of the custom SVG icon for the menu.
	 *
	 * @return string The URL to the SVG icon.
	 */
	public function get_custom_svg_icon() {
		// Assuming the SVG icon is saved in your plugin's 'assets' directory.
		return ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/icon.svg';
	}

	/**
	 * Registers the top-level admin menu page and adds a submenu page for 'OneTap'.
	 *
	 * This function uses the WordPress add_menu_page() and add_submenu_page() functions to create
	 * the necessary admin menu structure. It also sets up the callback functions for
	 * the top-level and submenu pages.
	 *
	 * @return void
	 */
	public function register_admin_menu_page() {
		add_menu_page(
			__( 'OneTap Pro', 'accessibility-plugin-onetap-pro' ), // Page title.
			__( 'OneTap Pro', 'accessibility-plugin-onetap-pro' ), // Menu title.
			'manage_options', // Capability required.
			'apop-settings', // Menu slug.
			array( $this, 'callback_template_for_settings' ), // Callback function.
			ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/menu.svg', // URL icon SVG.
			30 // $position.
		);

		add_submenu_page(
			'apop-settings', // Parent menu slug.
			__( 'Settings', 'accessibility-plugin-onetap-pro' ), // Page title.
			__( 'Settings', 'accessibility-plugin-onetap-pro' ), // Menu title.
			'manage_options', // Capability required.
			'apop-settings', // Menu slug.
			array( $this, 'callback_template_for_settings' ) // Callback function.
		);

		add_submenu_page(
			'apop-settings', // Parent menu slug.
			__( 'Modules', 'accessibility-plugin-onetap-pro' ), // Page title.
			__( 'Modules', 'accessibility-plugin-onetap-pro' ), // Menu title.
			'manage_options', // Capability required.
			'apop-modules', // Menu slug.
			array( $this, 'callback_template_for_modules' ) // Callback function.
		);
	}

	/**
	 * Loads the template for the 'Settings' menu page in the plugin.
	 *
	 * This function constructs the path to the template file located
	 * in the plugin directory and includes it if it exists.
	 */
	public function callback_template_for_settings() {
		// Define the path to the template file.
		$template_path = plugin_dir_path( __FILE__ ) . 'partials/settings.php';

		// Check if the template file exists.
		if ( file_exists( $template_path ) ) {
			// Include the template file if it exists.
			include $template_path;
		}
	}

	/**
	 * Loads the template for the 'Modules' menu page in the plugin.
	 *
	 * This function constructs the path to the template file located
	 * in the plugin directory and includes it if it exists.
	 */
	public function callback_template_for_modules() {
		// Define the path to the template file.
		$template_path = plugin_dir_path( __FILE__ ) . 'partials/modules.php';

		// Check if the template file exists.
		if ( file_exists( $template_path ) ) {
			// Include the template file if it exists.
			include $template_path;
		}
	}

	/**
	 * Remove notifications.
	 */
	public function hide_notifications_for_onetap_page() {
		if ( is_admin() && get_admin_page_parent() === 'apop-settings' ) {
			remove_all_actions( 'admin_notices' );
		}
	}

	/**
	 * Initialize the admin settings.
	 *
	 * This method sets the sections and fields for the settings API and initializes them.
	 *
	 * @return void
	 */
	public function admin_init() {

		// Set the settings api.
		$this->settings_api->set_sections( $this->get_settings_api_sections() );
		$this->settings_api->set_fields( $this->get_settings_api_fields() );

		// Initialize settings api.
		$this->settings_api->admin_init();
	}

	/**
	 * Retrieve the settings sections for the plugin.
	 *
	 * This method returns an array of sections used in the settings API.
	 * Each section contains an ID and a title.
	 *
	 * @return array The array of settings sections.
	 */
	public function get_settings_api_sections() {
		$sections = array(
			array(
				'id'    => 'apop_settings',
				'title' => __( 'Settings', 'accessibility-plugin-onetap-pro' ),
			),
			array(
				'id'    => 'apop_modules',
				'title' => __( 'Modules', 'accessibility-plugin-onetap-pro' ),
			),
		);
		return $sections;
	}

	/**
	 * Sanitize and validate the license key.
	 *
	 * This function takes a new license key as input, checks if it differs from
	 * the old license key stored in the database. If the keys are different, it
	 * deletes the old license key option, indicating that the plugin needs to
	 * be reactivated with the new license.
	 *
	 * @param string $value The new license key to be sanitized.
	 * @return string The sanitized license key.
	 */
	public function sanitize_license( $value ) {
		// Retrieve the old license key from the database.
		$old = Onetap_Pro_Config::get_setting( 'license' );

		// Check if an old license key exists and if it differs from the new one.
		if ( $old && $old !== $value ) {
			// Delete the old license setting to force reactivation.
			delete_option( 'apop_settings' )['license'];
		}

		// Sanitize the new license key before returning it.
		return sanitize_text_field( $value );
	}

	/**
	 * Returns all the settings fields
	 *
	 * @return array settings fields
	 */
	public function get_settings_api_fields() {
		$settings_fields = array(
			'apop_settings' => array(
				array(
					'name'              => 'icons',
					'label'             => __( 'Icons', 'accessibility-plugin-onetap-pro' ),
					'label_checkbox'    => __( 'Icons', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Design', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Customize your accessibility button’s color, icon, and size to match your brand.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/tutorial/customize-the-toolbar-icon/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'type'              => 'radio',
					'callback'          => 'callback_template_radio_icons',
					'default'           => Onetap_Pro_Config::get_setting( 'icons' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'size',
					'label'             => __( 'Size', 'accessibility-plugin-onetap-pro' ),
					'label_checkbox'    => __( 'Size', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'type'              => 'radio',
					'callback'          => 'callback_template_radio_size',
					'default'           => Onetap_Pro_Config::get_setting( 'size' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'border',
					'label'             => __( 'Border', 'accessibility-plugin-onetap-pro' ),
					'label_checkbox'    => __( 'Border', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'type'              => 'radio',
					'callback'          => 'callback_template_radio_border',
					'default'           => Onetap_Pro_Config::get_setting( 'border' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'color',
					'label'             => __( 'Color', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Colors', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Set your own branding colors to personalize the plugin’s appearance.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/color-options/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'type'              => 'color',
					'callback'          => 'callback_template_color',
					'default'           => Onetap_Pro_Config::get_setting( 'color' ),
					'sanitize_callback' => 'sanitize_hex_color',
				),
				array(
					'name'              => 'position-top-bottom',
					'label'             => __( 'Top/bottom position (px)', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Position', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Adjust the position of widgets to fit your layout preferences.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/position/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_top_bottom',
					'default'           => Onetap_Pro_Config::get_setting( 'position_top_bottom' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'position-left-right',
					'label'             => __( 'Left/right position (px)', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_left_right',
					'default'           => Onetap_Pro_Config::get_setting( 'position_left_right' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'widge-position',
					'label'             => __( 'Widget Position', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'type'              => 'select',
					'callback'          => 'callback_template_widget_position',
					'default'           => Onetap_Pro_Config::get_setting( 'widget_position' ),
					'sanitize_callback' => 'sanitize_text_field',
					'options'           => array(
						'middle-right' => __( 'Middle right', 'accessibility-plugin-onetap-pro' ),
						'middle-left'  => __( 'Middle left', 'accessibility-plugin-onetap-pro' ),
						'bottom-right' => __( 'Bottom right', 'accessibility-plugin-onetap-pro' ),
						'bottom-left'  => __( 'Bottom left', 'accessibility-plugin-onetap-pro' ),
						'top-right'    => __( 'Top right', 'accessibility-plugin-onetap-pro' ),
						'top-left'     => __( 'Top left', 'accessibility-plugin-onetap-pro' ),
					),
				),
				array(
					'name'              => 'position-top-bottom-tablet',
					'label'             => __( 'Top/bottom position (px)', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Position', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Adjust the position of widgets to fit your layout preferences.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/position/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_top_bottom',
					'default'           => Onetap_Pro_Config::get_setting( 'position_top_bottom_tablet' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'position-left-right-tablet',
					'label'             => __( 'Left/right position (px)', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_left_right',
					'default'           => Onetap_Pro_Config::get_setting( 'position_left_right_tablet' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'widge-position-tablet',
					'label'             => __( 'Widget Position', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'type'              => 'select',
					'callback'          => 'callback_template_widget_position',
					'default'           => Onetap_Pro_Config::get_setting( 'widget_position_tablet' ),
					'sanitize_callback' => 'sanitize_text_field',
					'options'           => array(
						'middle-right' => __( 'Middle right', 'accessibility-plugin-onetap-pro' ),
						'middle-left'  => __( 'Middle left', 'accessibility-plugin-onetap-pro' ),
						'bottom-right' => __( 'Bottom right', 'accessibility-plugin-onetap-pro' ),
						'bottom-left'  => __( 'Bottom left', 'accessibility-plugin-onetap-pro' ),
						'top-right'    => __( 'Top right', 'accessibility-plugin-onetap-pro' ),
						'top-left'     => __( 'Top left', 'accessibility-plugin-onetap-pro' ),
					),
				),
				array(
					'name'              => 'position-top-bottom-mobile',
					'label'             => __( 'Top/bottom position (px)', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Position', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Adjust the position of widgets to fit your layout preferences.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/position/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_top_bottom',
					'default'           => Onetap_Pro_Config::get_setting( 'position_top_bottom_mobile' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'position-left-right-mobile',
					'label'             => __( 'Left/right position (px)', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'min'               => 0,
					'max'               => 1000,
					'step'              => 1,
					'type'              => 'number',
					'callback'          => 'callback_template_number_left_right',
					'default'           => Onetap_Pro_Config::get_setting( 'position_left_right_mobile' ),
					'sanitize_callback' => 'absint',
				),
				array(
					'name'              => 'widge-position-mobile',
					'label'             => __( 'Widget Position', 'accessibility-plugin-onetap-pro' ),
					'desc'              => '',
					'type'              => 'select',
					'callback'          => 'callback_template_widget_position',
					'default'           => Onetap_Pro_Config::get_setting( 'widget_position_mobile' ),
					'sanitize_callback' => 'sanitize_text_field',
					'options'           => array(
						'middle-right' => __( 'Middle right', 'accessibility-plugin-onetap-pro' ),
						'middle-left'  => __( 'Middle left', 'accessibility-plugin-onetap-pro' ),
						'bottom-right' => __( 'Bottom right', 'accessibility-plugin-onetap-pro' ),
						'bottom-left'  => __( 'Bottom left', 'accessibility-plugin-onetap-pro' ),
						'top-right'    => __( 'Top right', 'accessibility-plugin-onetap-pro' ),
						'top-left'     => __( 'Top left', 'accessibility-plugin-onetap-pro' ),
					),
				),
				array(
					'name'              => 'language',
					'label'             => __( 'Default Language', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'Language', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Choose your preferred language for the plugin’s interface.', 'accessibility-plugin-onetap-pro' ) . ' <a href="https://wponetap.com/language-options/" target="_blank">' . __( 'See Documentation', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'type'              => 'select',
					'callback'          => 'callback_template_language',
					'default'           => Onetap_Pro_Config::get_setting( 'language' ),
					'sanitize_callback' => 'sanitize_text_field',
					'options'           => array(
						'en'    => __( 'English', 'accessibility-plugin-onetap-pro' ),
						'de'    => __( 'Deutsch', 'accessibility-plugin-onetap-pro' ),
						'es'    => __( 'Español', 'accessibility-plugin-onetap-pro' ),
						'fr'    => __( 'Français', 'accessibility-plugin-onetap-pro' ),
						'it'    => __( 'Italiano', 'accessibility-plugin-onetap-pro' ),
						'pl'    => __( 'Polski', 'accessibility-plugin-onetap-pro' ),
						'se'    => __( 'Svenska', 'accessibility-plugin-onetap-pro' ),
						'fi'    => __( 'Suomi', 'accessibility-plugin-onetap-pro' ),
						'pt'    => __( 'Português', 'accessibility-plugin-onetap-pro' ),
						'ro'    => __( 'Română', 'accessibility-plugin-onetap-pro' ),
						'si'    => __( 'Slovenščina', 'accessibility-plugin-onetap-pro' ),
						'sk'    => __( 'Slovenčina', 'accessibility-plugin-onetap-pro' ),
						'nl'    => __( 'Nederlands', 'accessibility-plugin-onetap-pro' ),
						'dk'    => __( 'Dansk', 'accessibility-plugin-onetap-pro' ),
						'gr'    => __( 'Ελληνικά', 'accessibility-plugin-onetap-pro' ),
						'cz'    => __( 'Čeština', 'accessibility-plugin-onetap-pro' ),
						'hu'    => __( 'Magyar', 'accessibility-plugin-onetap-pro' ),
						'lt'    => __( 'Lietuvių', 'accessibility-plugin-onetap-pro' ),
						'lv'    => __( 'Latviešu', 'accessibility-plugin-onetap-pro' ),
						'ee'    => __( 'Eesti', 'accessibility-plugin-onetap-pro' ),
						'hr'    => __( 'Hrvatski', 'accessibility-plugin-onetap-pro' ),
						'ie'    => __( 'Gaeilge', 'accessibility-plugin-onetap-pro' ),
						'bg'    => __( 'Български', 'accessibility-plugin-onetap-pro' ),
						'no'    => __( 'Norsk', 'accessibility-plugin-onetap-pro' ),
						'tr'    => __( 'Türkçe', 'accessibility-plugin-onetap-pro' ),
						'id'    => __( 'Bahasa Indonesia', 'accessibility-plugin-onetap-pro' ),
						'pt-br' => __( 'Português (Brasil)', 'accessibility-plugin-onetap-pro' ),
						'ja'    => __( '日本語', 'accessibility-plugin-onetap-pro' ),
						'ko'    => __( '한국어', 'accessibility-plugin-onetap-pro' ),
						'zh'    => __( '简体中文', 'accessibility-plugin-onetap-pro' ),
						'ar'    => __( 'العربية', 'accessibility-plugin-onetap-pro' ),
						'ru'    => __( 'Русский', 'accessibility-plugin-onetap-pro' ),
						'hi'    => __( 'हिन्दी', 'accessibility-plugin-onetap-pro' ),
						'uk'    => __( 'Українська', 'accessibility-plugin-onetap-pro' ),
						'sr'    => __( 'Srpski', 'accessibility-plugin-onetap-pro' ),
					),
				),
				array(
					'name'              => 'hide-powered-by-onetap',
					'label'             => __( 'Hide "Powered by OneTap"', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Easily remove the "Powered by OneTap" text from the sidebar on the frontend of your website.', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/hide-powered-by-onetap.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'hide_powered_by_onetap' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'license',
					'label'             => __( 'Enter License key', 'accessibility-plugin-onetap-pro' ),
					'site_title'        => __( 'License Key', 'accessibility-plugin-onetap-pro' ),
					'site_description'  => __( 'Enter your license key to unlock Pro features and gain access to advanced functionality.', 'accessibility-plugin-onetap-pro' ) .
					' <a target="_blank" href="' . esc_url( 'https://wponetap.com/' ) . '">' . __( 'Get a Pro License here', 'accessibility-plugin-onetap-pro' ) . '</a>',
					'desc'              => '',
					'placeholder'       => __( 'XXXXX-XXXXX-XXXXX-XXXXX', 'accessibility-plugin-onetap-pro' ),
					'type'              => 'text',
					'callback'          => 'callback_template_license',
					'default'           => Onetap_Pro_Config::get_setting( 'license' ),
					'sanitize_callback' => array( $this, 'sanitize_license' ),
				),
			),
			'apop_modules'  => array(
				array(
					'name'              => 'accessibility-profiles-title',
					'group_title'       => __( 'Accessibility Profiles', 'accessibility-plugin-onetap-pro' ),
					'group_description' => __( 'Smart solutions to enhance accessibility and improve overall usability.', 'accessibility-plugin-onetap-pro' ),
					'type'              => 'text',
					'callback'          => 'callback_template_general_module_title',
					'anchor'            => 'anchorAccessibilityProfiles',
					'default'           => true,
				),
				array(
					'name'              => 'accessibility-profiles',
					'label'             => __( 'Accessibility Profiles', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Ready-to-use accessibility profiles for different user needs.', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/accessibility-profiles.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'accessibility_profiles' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'content-modules-title',
					'group_title'       => __( 'Content Modules', 'accessibility-plugin-onetap-pro' ),
					'group_description' => __( 'Versatile tools to customize and enhance overall accessibility and usability.', 'accessibility-plugin-onetap-pro' ),
					'type'              => 'text',
					'callback'          => 'callback_template_general_module_title',
					'anchor'            => 'anchorContentModules',
					'default'           => true,
				),
				array(
					'name'              => 'bigger-text',
					'label'             => __( 'Bigger Text', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Easily enlarge text for improved readability and accessibility for all', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/bigger-text.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'bigger_text' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'cursor',
					'label'             => __( 'Cursor', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Enhance visibility with a larger cursor for easier navigation and control', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/cursor.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'cursor' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'line-height',
					'label'             => __( 'Line Height', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Adjust line spacing for better readability and improved text clarity', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/line-height.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'line_height' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'letter-spacing',
					'label'             => __( 'Letter Spacing', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Adjust letter spacing for improved readability', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/letter-spacing.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'letter_spacing' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'readable-font',
					'label'             => __( 'Readable Font', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Switch to a clearer, easy-to-read font for improved text accessibility', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/readable-font.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'readable_font' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'dyslexic-font',
					'label'             => __( 'Dyslexic Font', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Use a specialized font designed to enhance readability for dyslexic users', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/dyslexic-font.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'dyslexic_font' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'text-align',
					'label'             => __( 'Text Align', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Adjust text alignment for improved structure and readability', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/text-align.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'text_align' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'text-magnifier',
					'label'             => __( 'Text Magnifier', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Magnify selected text for enhanced readability and accessibility', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/text-magnifier.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'text_magnifier' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'highlight-links',
					'label'             => __( 'Highlight Links', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Easily identify clickable links with visual enhancements for better navigation', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/highlight-links.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'highlight_links' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'     => 'content-modules-title-save-changes',
					'type'     => 'text',
					'callback' => 'callback_template_save_changes',
				),
				array(
					'name'              => 'color-modules',
					'group_title'       => __( 'Colors', 'accessibility-plugin-onetap-pro' ),
					'group_description' => __( 'Options to adjust color settings for improved visibility and comfort.', 'accessibility-plugin-onetap-pro' ),
					'type'              => 'text',
					'callback'          => 'callback_template_general_module_title',
					'anchor'            => 'anchorModulesColors',
					'default'           => true,
				),
				array(
					'name'              => 'invert-colors',
					'label'             => __( 'Invert Colors', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Swap text and background colors for better contrast and readability', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/invert-colors.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'invert_colors' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'brightness',
					'label'             => __( 'Brightness', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Adjust screen brightness to reduce glare and enhance text visibility', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/brightness.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'brightness' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'contrast',
					'label'             => __( 'Contrast', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Fine-tune color contrast for clearer text and improved readability', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/contrast.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'contrast' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'grayscale',
					'label'             => __( 'Grayscale', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Convert display to black and white for reduced visual clutter and better focus', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/grayscale.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'grayscale' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'saturation',
					'label'             => __( 'Saturation', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Adjust color saturation for a more vivid or subdued visual experience', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/saturation.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'saturation' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'     => 'color-modules-save-changes',
					'type'     => 'text',
					'callback' => 'callback_template_save_changes',
				),
				array(
					'name'              => 'orientation-modules',
					'group_title'       => __( 'Orientation', 'accessibility-plugin-onetap-pro' ),
					'group_description' => __( 'Tools to enhance ease of movement and accessibility across the site.', 'accessibility-plugin-onetap-pro' ),
					'type'              => 'text',
					'callback'          => 'callback_template_general_module_title',
					'anchor'            => 'anchorOrientation',
					'default'           => true,
				),
				array(
					'name'              => 'reading-line',
					'label'             => __( 'Reading Line', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Highlight the current line to track reading progress and improve focus', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/reading-line.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'reading_line' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'keyboard-navigation',
					'label'             => __( 'Keyboard Navigation', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Navigate the site using only the keyboard', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/keyboard-navigation.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'keyboard_navigation' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'highlight-titles',
					'label'             => __( 'Highlight Titles', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Highlight titles for better recognition', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/highlight-titles.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'highlight_titles' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'reading-mask',
					'label'             => __( 'Reading Mask', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Use a mask to focus on a specific area of text and reduce distractions', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/reading-mask.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'reading_mask' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'hide-images',
					'label'             => __( 'Hide Images', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Remove distracting images for a cleaner, more focused browsing experience', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/hide-images.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'hide_images' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'highlight-all',
					'label'             => __( 'Highlight All', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Highlight elements when hovered over', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/highlight-all.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'highlight_all' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'read-page',
					'label'             => __( 'Read Page', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Automatically read aloud the page content for hands-free accessibility', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/read-page.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'read_page' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'mute-sounds',
					'label'             => __( 'Mute Sounds', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Easily mute website sounds for a distraction-free browsing experience', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/mute-sounds.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'mute_sounds' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'              => 'stop-animations',
					'label'             => __( 'Stop Animations', 'accessibility-plugin-onetap-pro' ),
					'desc'              => __( 'Disable animations to reduce distractions', 'accessibility-plugin-onetap-pro' ),
					'icon'              => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PLUGINS_URL . 'assets/images/admin/stop-animations.svg',
					'type'              => 'checkbox',
					'status'            => false,
					'callback'          => 'callback_template_checkbox',
					'default'           => Onetap_Pro_Config::get_module( 'stop_animations' ),
					'sanitize_callback' => 'sanitize_text_field',
				),
				array(
					'name'     => 'orientation-modules-save-changes',
					'type'     => 'text',
					'callback' => 'callback_template_save_changes',
				),
			),
		);

		return $settings_fields;
	}

	/**
	 * Get all the pages
	 *
	 * @return array page names with key value pairs
	 */
	public function get_pages() {
		$pages         = get_pages();
		$pages_options = array();
		if ( $pages ) {
			foreach ( $pages as $page ) {
				$pages_options[ $page->ID ] = $page->post_title;
			}
		}

		return $pages_options;
	}

	/**
	 * Handles the plugin updater functionality.
	 *
	 * This method checks if the current user has the appropriate capabilities
	 * or if the code is being run during a cron job for privileged users.
	 * It retrieves the license key from the database and initializes the
	 * plugin updater using the Easy Digital Downloads (EDD) updater class.
	 *
	 * @return void
	 */
	public function initialize_plugin_updater() {
		// To support auto-updates, this needs to run during the wp_version_check cron job for privileged users.
		$doing_cron = defined( 'DOING_CRON' ) && DOING_CRON;

		// Check if the current user has the capability to manage options or if this is a cron job.
		if ( ! current_user_can( 'manage_options' ) && ! $doing_cron ) {
			return; // Exit if the user does not have permission and it's not a cron job.
		}

		// Retrieve our settings from the DB.
		$settings = get_option( 'apop_settings' );

		// Check if the settings are an array and contain the 'license' key.
		$license_key = '';
		if ( is_array( $settings ) && isset( $settings['license'] ) ) {
			$license_key = trim( $settings['license'] );
		}

		// Setup the updater.
		$edd_updater = new Onetap_Plugin_Updater(
			ACCESSIBILITY_PLUGIN_ONETAP_PRO_STORE_URL,
			__FILE__,
			array(
				'version' => ACCESSIBILITY_PLUGIN_ONETAP_PRO_VERSION, // Current version number.
				'license' => $license_key, // License key retrieved from the database.
				'item_id' => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PRODUCT_ID, // ID of the product.
				'author'  => 'OneTap', // Author of this plugin.
				'beta'    => false, // Indicates whether this is a beta version.
			)
		);
	}

	/**
	 * Toggle the license activation state for the accessibility plugin.
	 *
	 * This function activates or deactivates the license based on the toggle
	 * setting in the plugin options. It communicates with the Easy Digital
	 * Downloads (EDD) server to update the license status.
	 *
	 * @return void
	 */
	public function toggle_license() {
		// Retrieve the plugin settings from the database.
		$settings = get_option( 'apop_settings' );

		// Get the license key from the settings; trim any whitespace.
		$license = isset( $settings['license'] ) ? trim( $settings['license'] ) : '';

		// Check if the activation button was pressed.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['onetap_license_activate'] ) ) {
			// Set up API parameters for activating the license.
			$api_params = $this->prepare_api_params( 'activate_license', $license );

			// Send the request to activate the license.
			$response = wp_remote_post(
				ACCESSIBILITY_PLUGIN_ONETAP_PRO_STORE_URL, // EDD store URL.
				array(
					'timeout'   => 15, // Request timeout in seconds.
					'sslverify' => true, // Ensure SSL verification for secure communication.
					'body'      => $api_params, // Body of the request containing parameters.
				)
			);

			// Process the response to determine if the activation was successful.
			$this->handle_license_response( $response, 'valid' );
		}

		// Check if the deactivation button was pressed.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['onetap_license_deactivate'] ) ) {
			// Set up API parameters for deactivating the license.
			$api_params = $this->prepare_api_params( 'deactivate_license', $license );

			// Send the request to deactivate the license.
			$response = wp_remote_post(
				ACCESSIBILITY_PLUGIN_ONETAP_PRO_STORE_URL, // EDD store URL.
				array(
					'timeout'   => 15, // Request timeout in seconds.
					'sslverify' => true, // Ensure SSL verification for secure communication.
					'body'      => $api_params, // Body of the request containing parameters.
				)
			);

			// Process the response to determine if the deactivation was successful.
			$this->handle_license_response( $response, 'invalid', true );
		}
	}

	/**
	 * Prepare API parameters for license activation or deactivation.
	 *
	 * @param string $action The action to perform (activate or deactivate).
	 * @param string $license The license key to activate or deactivate.
	 * @return array The prepared API parameters.
	 */
	private function prepare_api_params( $action, $license ) {
		return array(
			'edd_action'  => $action,
			'license'     => $license, // License key to activate or deactivate.
			'item_id'     => ACCESSIBILITY_PLUGIN_ONETAP_PRO_PRODUCT_ID, // Product ID.
			'item_name'   => rawurlencode( ACCESSIBILITY_PLUGIN_ONETAP_PRO_PRODUCT_NAME ), // Encoded product name.
			'url'         => esc_url( home_url() ), // Site URL, sanitized for security.
			'environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'production', // Environment type.
		);
	}

	/**
	 * Handle the license response from the EDD server.
	 *
	 * @param WP_Error|array $response The response from the EDD server.
	 * @param string         $valid_status The status to set if successful (valid/invalid).
	 * @param bool           $is_deactivation Whether this call is for deactivation.
	 * @return void
	 */
	private function handle_license_response( $response, $valid_status, $is_deactivation = false ) {
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( $license_data && $license_data->success ) {
			// Update the license status in the database.
			if ( $is_deactivation ) {
				delete_option( 'onetap_license_status' ); // Remove the status on successful deactivation.
			} else {
				update_option( 'onetap_license_status', $valid_status ); // Set status to valid on successful activation.
			}
		}
	}
}
